function ductsolve(geom,dim,spatial1,spatial2,a_mx,omega,mach,s_d,pfwd)

% spatial1 is alpha_mx in 2D, and also in 3D if spatial2 is false
% otherwise spatial1 and spatial2 are p_mx and q_mx in 3D
% s_d allows for specification of downstream arclength
% pfwd is true for specification of p+ at inlet, false for p

addpath(genpath('.'))

%% DUCT GEOMETRY
disp('creating duct...')
geom        = str2func(strcat(geom,num2str(dim),'D'));
geom        = geom();
s_o         = geom.L;
geomname    = geom().name;
geomparams  = geom().geomparams;
geom.inner  = false;

% geometry functions
kappa_func  = geom().kappa;
if dim == 2
    X_func      = geom().X;
    spatial2    = false;
    qzvar       = false;
elseif dim == 3
    R_func      = geom().R;
    tau_func    = geom().tau;
    qyvar       = geom().qyvar;
    qzvar       = geom().qzvar;
end

if s_d == false
    s_d = s_o;
end


%% NUMERICS
disp('setting parameters...')
% choose solver
fixedstep = false;
magmob = false;
if fixedstep == true
    delta = 10^-3;
    numparams.delta     = delta;
end

% p_alpha can be raised above zero for antisymmetric modes in 2D and
% non-plane inlet conditions in 3D
p_alpha = 0;

if dim == 2
    alpha_mx = spatial1;
    modes = alphagen2D(alpha_mx);
elseif dim == 3
    if spatial2 == false
        alpha_mx = spatial1;
        if qyvar == false && p_alpha == 0
            modes = pqxigen3D(0,alpha_mx,0);
        elseif qzvar == false
            modes = pqxigen3D(alpha_mx,alpha_mx,0);
            modes = modes(1:alpha_mx+1,:);
        elseif qzvar == true
            modes = pqxigen3D(alpha_mx,alpha_mx,1);
            modes = modes(1:alpha_mx+1,:);
        end
        p_mx = max(modes(:,3));
        q_mx = max(modes(:,4));
    else
        p_mx = spatial1;
        q_mx = spatial2;
        if qzvar == false
            modes = pqxigen3D(p_mx,q_mx,0);
            alpha_mx = (p_mx+1)*q_mx;
        elseif qzvar == true
            modes = pqxigen3D(p_mx,q_mx,1);
            alpha_mx = (2*p_mx+1)*q_mx;
        end
    end
end
if dim == 3
    p_mx    = max(modes(:,3));
    q_mx    = max(modes(:,4));
    xi_mx   = max(modes(:,5));
end

if dim == 2
    alpha_entry = mod(p_alpha,2);
elseif dim == 3
    alpha_entry = 0;
    while modes(alpha_entry+1,3) ~= p_alpha && alpha_entry < alpha_mx
        alpha_entry = alpha_entry+1;
    end
    if alpha_entry == alpha_mx && modes(alpha_entry+1,3) ~= p_alpha
        disp('error: more modes required')
    end
end

%constants
gamma0      = 1.4;
beta0       = (gamma0+1)/2;
numvisc     = 1;
if mach == false
    nonlinear   = false;
    mach        = 0.1;
else
    nonlinear = true;
end

numparams.alpha_mx  = alpha_mx;
if dim == 3
    numparams.p_mx      = p_mx;
    numparams.q_mx      = q_mx;
    numparams.spatial2  = spatial2;
end
numparams.alpha_entry = alpha_entry;
numparams.a_mx      = a_mx;
numparams.nonlinear = nonlinear;
numparams.modes     = modes;
numparams.omega     = omega;
numparams.beta0     = beta0;
numparams.mach      = mach;
numparams.numvisc   = numvisc;
numparams.s_d       = s_d;
numparams.pfwd      = pfwd;

% choose what we save
lightload = true;
numparams.lightload = lightload;


%% LINEAR MATRIX DEFINITIONS
disp('calculating Bessel matrices...')
modegen = str2func(strcat('sfgen',num2str(dim),'D'));
if dim == 2
    load('current2Dlmats.mat','lmats','alpha_mx_current');
    if alpha_mx_current ~= alpha_mx
        lmats = modegen(modes);
        alpha_mx_current = alpha_mx;
        save('current2Dlmats.mat','lmats','alpha_mx_current')
    end
elseif dim == 3
    load('current3Dlmats.mat','lmats','alpha_mx_current',...
        'p_mx_current','q_mx_current','xi_mx_current')
    if alpha_mx_current ~= alpha_mx || p_mx_current ~= p_mx ...
            || q_mx_current ~= q_mx || xi_mx_current ~= xi_mx
        lmats = modegen(modes);
        alpha_mx_current    = alpha_mx;
        p_mx_current        = p_mx;
        q_mx_current        = q_mx;
        xi_mx_current       = xi_mx;
        save('current3Dlmats.mat','lmats','alpha_mx_current',...
            'p_mx_current','q_mx_current','xi_mx_current')
    end
end

if nonlinear == false
    nlmats      = 0;
else
    modegen = str2func(strcat('calgen',num2str(dim),'D'));
    if dim == 2
        load('current2Dnlmats.mat','nlmats','alpha_mx_current');
        if alpha_mx_current ~= alpha_mx
            nlmats = modegen(modes);
            alpha_mx_current = alpha_mx;
            save('current2Dnlmats.mat','nlmats','alpha_mx_current')
        end
    elseif dim == 3
        load('current3Dnlmats.mat','nlmats','alpha_mx_current',...
            'p_mx_current','q_mx_current','xi_mx_current')
        if alpha_mx_current ~= alpha_mx || p_mx_current ~= p_mx ...
                || q_mx_current ~= q_mx || xi_mx_current ~= xi_mx
            nlmats = modegen(modes);
            alpha_mx_current    = alpha_mx;
            p_mx_current        = p_mx;
            q_mx_current        = q_mx;
            xi_mx_current       = xi_mx;
            save('current3Dnlmats.mat','nlmats','alpha_mx_current',...
                'p_mx_current','q_mx_current','xi_mx_current')
        end
    end
end

charmats_o = evaluefinder(s_o,numparams,geom,lmats,nlmats,3);
Ysfpl_o_pos = charmats_o.Ysfpl_pos;


%% LINEAR ADMITTANCE SOLUTION
disp('solving linear admittance ODE...')
if magmob == false
    if fixedstep == false
        [s_Ysfsol_pos,Ysfsol_pos] ...
            = ode45(@(s,Ysf_pos) dYsf_ds_def(s,Ysf_pos,numparams,geom,...
            lmats),[s_o 0],Ysfpl_o_pos);
    else
        Ysfpl_o_pos = reshape(Ysfpl_o_pos,(alpha_mx+1)^2*a_mx,1);
        [Ysfsol_pos,s_Ysfsol_pos,~] = rk4ode(@(Ysf_pos,s) ...
            dYsf_ds_def(s,Ysf_pos,numparams,geom,lmats),@(Ysf_pos,ds) ...
            Ysf_pos,Ysfpl_o_pos,s_o,-delta,0,false);
    end
    method = 'linear';
    s_Ysfsol_pos = flipud(s_Ysfsol_pos);
    Ysfsol_pos = flipud(Ysfsol_pos);
else
    [s_Ysfsol_pos,Ysfsol_pos,norms] ...
        = Ysfmagnusmobius([s_o 0],Ysfpl_o_pos,numparams,geom,lmats);
    hold off
    plot(s_Ysfsol_pos,log10(norms.Zsf_posnorm))
    hold on
    plot(s_Ysfsol_pos,log10(norms.Ysf_posnorm))
    label1 = 'error in $$\mathsf{Z}^a$$';
    label2 = 'error in $$\mathsf{Y}^a$$';
    xlabel('$$s$$')
    ylabel('$$\log_{10}($$error$$)$$')
    labels = {label1;label2};
    if s_d == s_o
        method = 'cubic';
    else
        method = 'linear';
    end
end

if s_d > s_o
    %concatenate with characteristic admittance for interpolation
    s_Ysfsol_pos = cat(1,s_Ysfsol_pos,s_d);
    Ysfsol_pos ...
        =cat(1,Ysfsol_pos,reshape(Ysfpl_o_pos,[1,a_mx*(alpha_mx+1)^2]));
end

Ysfsol = zeros(length(s_Ysfsol_pos),(2*a_mx+1)*(alpha_mx+1)^2);
for a = 1:a_mx
    Ysfsol(:,(a_mx-a)*(alpha_mx+1)^2+1:(a_mx+1-a)*(alpha_mx+1)^2) ...
        = conj(Ysfsol_pos(:,(a-1)*(alpha_mx+1)^2+1:a*(alpha_mx+1)^2));
end
Ysfsol(:,(a_mx+1)*(alpha_mx+1)^2+1:(2*a_mx+1)*(alpha_mx+1)^2)...
    = Ysfsol_pos;

%for interpolation grids, capitalised S and subscript interpolant 
% quantity followed by purpose of this grid

S_Ysf_Ysfsol    = {s_Ysfsol_pos, 1:size(Ysfsol,2)};
Ysfint          = griddedInterpolant(S_Ysf_Ysfsol,Ysfsol,method);
geom.Ysfint     = Ysfint;

% S_Ysf_Ysftest   = {0:s_d/5:s_d, 1:(alpha_mx+1)^2*(2*a_mx+1)};
% permute(reshape(Ysfint(S_Ysf_Ysftest),[],alpha_mx+1,alpha_mx+1,...
%     2*a_mx+1),[2 3 4 1]);



%% NONLINEAR ADMITTANCE
if nonlinear == true
    disp('solving nonlinear admittance ODE...')
    Ycalpl_o_pos = charmats_o.Ycalpl_pos;
    if magmob == false
        if fixedstep == false
            [s_Ycalsol_pos,Ycalsol_pos] =ode45(@(s,YY) dYcal_ds_def(s,...
                YY,numparams,geom,lmats,nlmats),[s_o 0],Ycalpl_o_pos);
        else
            Ycalpl_o_pos = reshape(Ycalpl_o_pos,(alpha_mx+1)^3*a_mx...
                *(2*a_mx+1),1);
            [Ycalsol_pos,s_Ycalsol_pos,~] ...
                = rk4ode(@(Ycal_pos,s) dYcal_ds_def(s,Ycal_pos,...
                numparams,geom,lmats,nlmats),@(Ycal_pos,ds) Ycal_pos,...
                Ycalpl_o_pos,s_o,-delta,0,false);
        end

        s_Ycalsol_pos   = flipud(s_Ycalsol_pos);
        Ycalsol_pos     = flipud(Ycalsol_pos);
    else
        [s_Ycalsol_pos,Ycalsol_pos,norms] = Ycalmagnusmobius([s_o 0],...
            Ycalpl_o_pos,numparams,geom,lmats,nlmats);
        Ycalnorm = norms.Ycalnorm;
        Zcalnorm = norms.Zcalnorm;

        plot(s_Ycalsol_pos,log10(Zcalnorm))
        plot(s_Ycalsol_pos,log10(Ycalnorm))
        labels = cat(1,labels,'error in $$\mathcal{Z}^{ab}$$',...
            'error in $$\mathcal{Y}^{ab}$$');
        legend(labels)
    end
    %concatenate with characteristic admittance for interpolation
    if s_d > s_o
        s_Ycalsol_pos = cat(1,s_Ycalsol_pos,s_d);
        Ycalsol_pos = cat(1,Ycalsol_pos,reshape(Ycalpl_o_pos,...
            [1,a_mx*(2*a_mx+1)*(alpha_mx+1)^3]));
    end
    S_Ycal_Ycalsol  = {s_Ycalsol_pos,1:size(Ycalsol_pos,2)};
    Ycalint_pos     ...
            = griddedInterpolant(S_Ycal_Ycalsol,Ycalsol_pos,'linear');
    geom.Ycalint    = Ycalint_pos;

    % S_Ycal_Ycaltest = {0:s_d/5:s_d, 1:size(Ycalsol_pos,2)};
    % reshape(Ycalint_pos(S_Ycal_Ycaltest)',a_mx,2*a_mx+1,6,[]);
    % reshape(Ycalint_pos({9.8,1:size(Ycalsol_pos,2)}),a_mx,2*a_mx+1,[]);

end


%% PRESSURE SOLUTION
disp('solving pressure ODE...')
%incident wave longitudinal angle
theta_l     = 0;
if theta_l ~= 0 && dim == 2
    Xmn_func = geom().Xmn;
    Asf = lmats.Asf;
    matIC = (1+2*tan(theta_l)*Xmn_func(0)/X_func(0))*eye(alpha_mx+1)...
        +2*tan(theta_l)*Asf;
elseif theta_l ~= 0 && dim == 3
    %incident wave transverse angle
    theta_t     = 0;
    Pi_x2       = lmats.Pi_x2;
    Phi_cp      = lmats.Phi_cp;
    Phi_sp      = lmats.Phi_sp;
    matIC       = eye(alpha_mx+1) + tan(theta_l)*Pi_x2...
        .*(cos(theta_t)*Phi_cp + sin(theta_t)*Phi_sp);
else
    matIC = eye(alpha_mx+1);
end

if dim == 2
    Acs_i = sqrt(X_func(0));
elseif dim == 3
    Acs_i = sqrt(pi)*R_func(0);
end

Ppl_i_pos           = zeros(alpha_mx+1,a_mx);
Ppl_i_pos(:,1)      = Acs_i*mach*matIC(:,alpha_entry+1)/(2*1i);

charmats_i = evaluefinder(0,numparams,geom,lmats,nlmats,3);

smats_i = splittingdefs(0,numparams,geom,charmats_i);
Ssfpl_i_pos     = smats_i.Ssfpl_pos;
if nonlinear == true
    Ssfpl_i         = smats_i.Ssfpl;
    Scalpl_i_pos    = smats_i.Scalpl_pos;
end


if pfwd == false
    Pinput = Ppl_i_pos;
else
    P_i_pos = zeros(alpha_mx+1,a_mx);
    if nonlinear == false
        for a = 1:a_mx
            P_i_pos(:,a) = Ssfpl_i_pos(:,:,a)\Ppl_i_pos(:,a);
        end
    else
        Ppl_i = zeros(alpha_mx+1,2*a_mx+1);
        Ppl_i(:,a_mx+2:2*a_mx+1) = Ppl_i_pos(:,1:a_mx);
        Ppl_i(:,a_mx:-1:1)       = conj(Ppl_i_pos(:,1:a_mx));
        dot = ones(2*a_mx+1,1);
        Scalpl_i_inv_PP_pos = zeros(alpha_mx+1,a_mx,2*a_mx+1);
        for a = 1:a_mx
            for b = -a_mx+a:a_mx
                if b ~= 0 && b ~= a
                        Scalpl_i_inv_pos = -tensorprod(inv(...
                            Ssfpl_i(:,:,a+a_mx+1)),tensorprod(...
                            tensorprod(Scalpl_i_pos(:,:,:,a,b+a_mx+1),...
                            inv(Ssfpl_i(:,:,a-b+a_mx+1)),2,1),...
                            inv(Ssfpl_i(:,:,b+a_mx+1)),2,1),2,1);
                        Scalpl_i_inv_PP_pos(:,a,b+a_mx+1) ...
                            = tensorprod(tensorprod(Scalpl_i_inv_pos,...
                            Ppl_i(:,a-b+a_mx+1),2,1),...
                            Ppl_i(:,b+a_mx+1),2,1);
                end
            end
            P_i_pos(:,a) = Ssfpl_i(:,:,a+a_mx+1)\Ppl_i_pos(:,a) ...
                + reshape(Scalpl_i_inv_PP_pos(:,a,:),alpha_mx+1,...
                2*a_mx+1)*dot;
        end
    end
    Pinput = P_i_pos;
end
if magmob == false
    if fixedstep == false
        [s_Psol,Psol_pos]   = ode45(@(s,P_pos) dPds_def(s,P_pos,...
            numparams,geom,lmats,nlmats),[0 s_d],Pinput);
    else
        Pinput = reshape(Pinput,(alpha_mx+1)*a_mx,1);
        [Psol_pos,s_Psol,~] ...
            = rk4ode(@(P_pos,s) dPds_def(s,P_pos,numparams,geom,...
            lmats,nlmats),@(P_pos,ds) P_pos,Pinput,0,delta,s_d,false);
    end
else
    [s_Psol,Psol_pos,norms] = Pmagnusmobius([0 s_d],...
        Pinput,numparams,geom,lmats,nlmats);
    plot(s_Psol,log10(norms.Pnorm_pos))
    labels = cat(1,labels,{'error in $$\mathbf{p}^{a}$$'});
    legend(labels)
end

P_sol = zeros(length(s_Psol),(2*a_mx+1)*(alpha_mx+1));
for a = 1:a_mx
    P_sol(:,(a_mx-a)*(alpha_mx+1)+1:(a_mx-a+1)*(alpha_mx+1))...
        = conj(Psol_pos(:,(a-1)*(alpha_mx+1)+1:a*(alpha_mx+1)));
end
P_sol(:,(a_mx+1)*(alpha_mx+1)+1:(2*a_mx+1)*(alpha_mx+1)) = Psol_pos;


%% PRESSURE SPLITTING
P_Psol      = permute(reshape(P_sol,[],alpha_mx+1,2*a_mx+1),[2 3 1]);
kappa_Psol  = zeros(length(s_Psol),1);
if dim == 2
    X_Psol      = zeros(length(s_Psol),1);
elseif dim == 3
    R_Psol      = zeros(length(s_Psol),1);
    tau_Psol    = zeros(length(s_Psol),1);
end
Ppl_Psol_pos    = zeros(alpha_mx+1,a_mx,length(s_Psol));
Pmn_Psol_pos    = zeros(alpha_mx+1,a_mx,length(s_Psol));

if nonlinear == false
    Ysfpl_pos       = zeros(alpha_mx+1,alpha_mx+1,a_mx,length(s_Psol));
    Ysfmn_pos       = zeros(alpha_mx+1,alpha_mx+1,a_mx,length(s_Psol));
else
    Ysfpl = zeros(alpha_mx+1,alpha_mx+1,2*a_mx+1,length(s_Psol));
    Ysfmn = zeros(alpha_mx+1,alpha_mx+1,2*a_mx+1,length(s_Psol));
    Ycalpl_pos      = zeros(alpha_mx+1,alpha_mx+1,alpha_mx+1,...
        a_mx,2*a_mx+1,length(s_Psol));
    Ycalmn_pos      = zeros(alpha_mx+1,alpha_mx+1,alpha_mx+1,...
        a_mx,2*a_mx+1,length(s_Psol));
end

for I_s = 1:length(s_Psol)
    kappa_Psol(I_s) = kappa_func(s_Psol(I_s));
    if dim == 2
        X_Psol(I_s)     = X_func(s_Psol(I_s));
        if I_s == 1
            charmats = charmats_i;
        elseif I_s > 1 && kappa_Psol(I_s) == kappa_Psol(I_s-1) ...
                && s_Psol(I_s) ~= s_o && X_Psol(I_s) == X_Psol(I_s-1)
        elseif s_Psol(I_s) == s_o
            charmats = charmats_o;
        else
            charmats = evaluefinder(s_Psol(I_s),numparams,geom,lmats,...
                nlmats,1);
        end
    elseif dim == 3
        R_Psol(I_s)     = R_func(s_Psol(I_s));
        tau_Psol(I_s)   = tau_func(s_Psol(I_s));
        if I_s == 1
            charmats = charmats_i;
        elseif I_s > 1 && kappa_Psol(I_s) == kappa_Psol(I_s-1) ...
                && s_Psol(I_s) ~= s_o && R_Psol(I_s) == R_Psol(I_s-1) ...
                && tau_Psol(I_s) == tau_Psol(I_s-1)
        elseif s_Psol(I_s) == s_o
            charmats = charmats_o;
        else
            charmats = evaluefinder(s_Psol(I_s),numparams,geom,lmats,...
                nlmats,1);
        end
    end
    if nonlinear == false
        Ysfpl_pos(:,:,:,I_s)    = charmats.Ysfpl_pos;
        Ysfmn_pos(:,:,:,I_s)    = charmats.Ysfmn_pos;
    else
        Ysfpl(:,:,:,I_s)            = charmats.Ysfpl;
        Ysfmn(:,:,:,I_s)            = charmats.Ysfmn;
        Ycalpl_pos(:,:,:,:,:,I_s)   = charmats.Ycalpl_pos;
        Ycalmn_pos(:,:,:,:,:,I_s)   = charmats.Ycalmn_pos;
    end
    smats = splittingdefs(s_Psol(I_s),numparams,geom,charmats);
    Ssfpl_pos = smats.Ssfpl_pos;
    Ssfmn_pos = smats.Ssfmn_pos;
    if nonlinear == false
        for a = 1:a_mx
            Ppl_Psol_pos(:,a,I_s) = Ssfpl_pos(:,:,a)...
                *P_Psol(:,a_mx+1+a,I_s);
            Pmn_Psol_pos(:,a,I_s) = Ssfmn_pos(:,:,a)...
                *P_Psol(:,a_mx+1+a,I_s);
        end
    else
        Scalpl_pos  = smats.Scalpl_pos;
        Scalmn_pos  = smats.Scalmn_pos;
        reshape(Scalmn_pos,a_mx,[]);
        for a = 1:a_mx
            ScalplPP_pos = zeros(alpha_mx+1,a_mx,2*a_mx+1);
            ScalmnPP_pos = zeros(alpha_mx+1,a_mx,2*a_mx+1);
            dot = ones(2*a_mx+1,1);
            for b = -a_mx+a:a_mx
                if b ~= 0 && b ~= a
                    ScalplPP_pos(:,a,b+a_mx+1) ...
                        = tensorprod(tensorprod(...
                        Scalpl_pos(:,:,:,a,b+a_mx+1),...
                        P_Psol(:,a-b+a_mx+1),2,1),...
                        P_Psol(:,b+a_mx+1),2,1);
                    ScalmnPP_pos(:,a,b+a_mx+1) ...
                        = tensorprod(tensorprod(...
                        Scalmn_pos(:,:,:,a,b+a_mx+1),...
                        P_Psol(:,a-b+a_mx+1),2,1),...
                        P_Psol(:,b+a_mx+1),2,1);
                end
            end
            Ppl_Psol_pos(:,a,I_s) = Ssfpl_pos(:,:,a)...
                *P_Psol(:,a_mx+1+a,I_s) + reshape(...
                ScalplPP_pos(:,a,:),alpha_mx+1,2*a_mx+1)*dot;
            Pmn_Psol_pos(:,a,I_s) = Ssfmn_pos(:,:,a)...
                *P_Psol(:,a_mx+1+a,I_s) + reshape(...
                ScalmnPP_pos(:,a,:),alpha_mx+1,2*a_mx+1)*dot;
        end
    end
end

Ppl_Psol = zeros(alpha_mx+1,2*a_mx+1,length(s_Psol));
Pmn_Psol = zeros(alpha_mx+1,2*a_mx+1,length(s_Psol));

Ppl_Psol(:,a_mx+2:2*a_mx+1,:) = Ppl_Psol_pos;
Pmn_Psol(:,a_mx+2:2*a_mx+1,:) = Pmn_Psol_pos;

Ppl_Psol(:,a_mx:-1:1,:) = conj(Ppl_Psol_pos);
Pmn_Psol(:,a_mx:-1:1,:) = conj(Pmn_Psol_pos);


Ppl_Psol =reshape(permute(Ppl_Psol,[3 1 2]),[],(alpha_mx+1)*(2*a_mx+1));
Pmn_Psol =reshape(permute(Pmn_Psol,[3 1 2]),[],(alpha_mx+1)*(2*a_mx+1));

S_P_Psol        = {s_Psol,1:(alpha_mx+1)*(2*a_mx+1)};
Pint            = griddedInterpolant(S_P_Psol,P_sol,method);
Pplint          = griddedInterpolant(S_P_Psol,Ppl_Psol,method);
Pmnint          = griddedInterpolant(S_P_Psol,Pmn_Psol,method);

if lightload == false
    if nonlinear == false
        Ysfpl = zeros(alpha_mx+1,alpha_mx+1,2*a_mx+1,length(s_Psol));
        Ysfmn = zeros(alpha_mx+1,alpha_mx+1,2*a_mx+1,length(s_Psol));
        for a = 1:a_mx
            Ysfpl(:,:,a_mx:-1:1,:) = conj(Ysfpl_pos);
            Ysfmn(:,:,a_mx:-1:1,:) = conj(Ysfmn_pos);
        end
        Ysfpl(:,:,a_mx+2:2*a_mx+1,:) = Ysfpl_pos;
        Ysfmn(:,:,a_mx+2:2*a_mx+1,:) = Ysfmn_pos;
    else
        S_P_Ycalpl      = {s_Psol,1:(alpha_mx+1)^3*a_mx*(2*a_mx+1)};
        Ycalplint       = griddedInterpolant(S_P_Ycalpl,permute(reshape(...
            Ycalpl_pos,(alpha_mx+1)^3*a_mx*(2*a_mx+1),[]),[2 1]),method);
        Ycalmnint       = griddedInterpolant(S_P_Ycalpl,permute(reshape(...
            Ycalmn_pos,(alpha_mx+1)^3*a_mx*(2*a_mx+1),[]),[2 1]),method);
    end

    S_P_Ysfpl       = {s_Psol,1:(alpha_mx+1)^2*(2*a_mx+1)};
    Ysfplint        = griddedInterpolant(S_P_Ysfpl,permute(...
        reshape(Ysfpl,(alpha_mx+1)^2*(2*a_mx+1),[]),[2 1]),method);
    Ysfmnint        = griddedInterpolant(S_P_Ysfpl,permute(...
        reshape(Ysfmn,(alpha_mx+1)^2*(2*a_mx+1),[]),[2 1]),method);
end


% s_Ptest         = 0:s_d/7:s_d;
% S_P_Ptest       = {s_Ptest,1:(alpha_mx+1)*(2*a_mx+1)};
% P_Ptest ...
%     = permute(reshape(Pint(S_P_Ptest),[],alpha_mx+1,2*a_mx+1),[2 3 1]);
% Ppl_Ptest ...
%     = permute(reshape(Pplint(S_P_Ptest),[],alpha_mx+1,2*a_mx+1),[2 3 1]);
% Pmn_Ptest ...
%     = permute(reshape(Pmnint(S_P_Ptest),[],alpha_mx+1,2*a_mx+1),[2 3 1]);


%% RESULTS
if spatial2 == false
    filename = strcat(geomname,sprintf('_alpha%03.f',alpha_mx));
else
    filename = strcat(geomname,sprintf('_p%03.f_q%03.f',p_mx,q_mx));
end
if nonlinear == true
    filename = strcat(filename,sprintf(...
        '_a%03.f_omega%06.4f_M%04.2f_s_d%07.4f',a_mx,omega,mach,s_d));
else
    filename = strcat(filename,sprintf(...
        '_a%03.f_omega%06.4f_linear_s_d%07.4f',a_mx,omega,s_d));
end
if pfwd == false
    filename = strcat(filename,'_ptot');
else
    filename = strcat(filename,'_pfwd');
end
if qzvar == true
    filename = strcat(filename,sprintf('_tauR%04.2f',...
        tau_func(s_o)*R_func(s_o)));
end
filename = strcat(filename,'.mat');

fprintf('%s\n\n', filename);
if lightload == true
    save(filename,'numparams','geomname','geomparams','Pint','Pplint',...
        'Pmnint','-v7.3')
else
    if nonlinear == false
        save(filename,'numparams','geomname','geomparams','Ysfint',...
            'Ysfplint','Ysfmnint','Pint','Pplint','Pmnint','-v7.3')
    else
        save(filename,'numparams','geomname','geomparams','Ysfint',...
            'Ysfplint','Ysfmnint','Ycalplint','Ycalmnint',...
            'Ycalint_pos','Pint','Pplint','Pmnint','-v7.3')
    end
end
end
