function nlmats = calgen2D(modes)

% generates calligraphic tensors for projected 2D duct acoustics equations,
% given mode table

%% NUMERICAL DATA
alpha_mx = length(modes(:,1)) - 1;


%% NONLINEAR MATRIX DEFINITIONS
% 2D nonlinear building-block matrices
XiXi        = zeros(alpha_mx+1,alpha_mx+1,alpha_mx+1);
XiXi_xi     = zeros(alpha_mx+1,alpha_mx+1,alpha_mx+1);
XiXi_d      = zeros(alpha_mx+1,alpha_mx+1,alpha_mx+1);
XiXi_b      = zeros(alpha_mx+1,alpha_mx+1,alpha_mx+1);
XiXi_b1     = zeros(alpha_mx+1,alpha_mx+1,alpha_mx+1);
XiXi_b0     = zeros(alpha_mx+1,alpha_mx+1,alpha_mx+1);

for gamma = 0:alpha_mx
    fprintf(1,'gamma = %3.0f\r', gamma);
    for beta = 0:alpha_mx
        for alpha = 0:alpha_mx
            if alpha + beta == gamma || abs(alpha - beta) == gamma
                XiXi(alpha+1,beta+1,gamma+1) = 0.5*modes(alpha+1,3)...
                    *modes(beta+1,3)/modes(gamma+1,3)*(...
                    eq(alpha+beta,gamma) + eq(abs(alpha-beta),gamma));
                XiXi_xi(alpha+1,beta+1,gamma+1) = 0.5*XiXi(alpha+1,...
                    beta+1,gamma+1);
            else
                XiXi_xi(alpha+1,beta+1,gamma+1) = (4*pi^2)^-1*...
                    modes(alpha+1,3)*modes(beta+1,3)*modes(gamma+1,3)...
                    *((-1)^(alpha+beta+gamma) - 1)...
                    *((alpha+beta+gamma)^-2 + (alpha+beta-gamma)^-2 ...
                    + (alpha-beta+gamma)^-2 + (alpha-beta-gamma)^-2);
                XiXi_d(alpha+1,beta+1,gamma+1) = 2^-1.5...
                    *alpha*modes(beta+1,3)*modes(gamma+1,3)...
                    *((-1)^(alpha + beta + gamma) - 1)...
                    *((alpha+beta-gamma)^-1 + (alpha-beta+gamma)^-1 ...
                    + (alpha-beta-gamma)^-1);
            end
            XiXi_b(alpha+1,beta+1,gamma+1) = modes(alpha+1,3)...
                *modes(beta+1,3)*modes(gamma+1,3)...
                *((-1)^(alpha+beta+gamma) - 1);
            XiXi_b1(alpha+1,beta+1,gamma+1) = modes(alpha+1,3)...
                *modes(beta+1,3)*modes(gamma+1,3)*(-1)^(alpha+beta+gamma);
            XiXi_b0(alpha+1,beta+1,gamma+1) = modes(alpha+1,3)...
                *modes(beta+1,3)*modes(gamma+1,3);
        end
    end
end

Ical        = XiXi;
Acal        = XiXi_xi;
Atilcal     = XiXi_d;
Abarcal     = XiXi_b;
Wplbarcal   = XiXi_b1;
Wmnbarcal   = XiXi_b0;

Lambdasf    = diag(modes(:,2));
IEcal       = 0.5*(permute(tensorprod(Ical,Lambdasf,2,1),[1 3 2]) ...
                + tensorprod(Ical,Lambdasf,3,1) ...
                - tensorprod(Lambdasf,Ical,2,1));
AEcal       = 0.5*(permute(tensorprod(Acal,Lambdasf,2,1),[1 3 2]) ...
                + tensorprod(Acal,Lambdasf,3,1) ...
                - tensorprod(Lambdasf,Acal,2,1)) + Atilcal - 0.5*Abarcal;
AtilEcal    = 0.5*(permute(tensorprod(Atilcal,Lambdasf,2,1),[1 3 2]) ...
                + tensorprod(Atilcal,Lambdasf,3,1) ...
                - tensorprod(Lambdasf,Atilcal,2,1) ...
                + tensorprod(Lambdasf^2,Abarcal,2,1));

nlmats.Ical     = Ical;
nlmats.Wplbarcal= Wplbarcal;
nlmats.Wmnbarcal= Wmnbarcal;
nlmats.Acal     = Acal;
nlmats.Atilcal  = Atilcal;
nlmats.IEcal    = IEcal;
nlmats.AEcal    = AEcal;
nlmats.AtilEcal = AtilEcal;

end
