function [k, itter, errs] = besscrosszero(nu, l, N, varargin)
    % BESSCROSSZERO the roots of the Bessel function cross products
    %
    %     k = BESSCROSSZERO(m, labmda, N) the Nth root of the Bessel
    %     function cross products J(m,x)*Y(m,labmda*x) - Y(m,x)*J(m,labmda*x)
    %
    %     k = BESSCROSSZERO(m, labmda, N, BC) the Nth root of the function,
    %     when, BC = 'DD',
    %             J(m,x)*Y(m,labmda*x) - Y(m,x)*J(m,labmda*x)
    %           BC ='DN':
    %             Y'(m,x)*J(m,labmda*x) - J'(m,x)*Y(m,labmda*x)
    %           BC ='ND':
    %             J(m,x)*Y'(m,labmda*x) - Y(m,x)*J'(m,labmda*x)
    %           BC ='NN':
    %             J'(m,x)*Y'(m,labmda*x) - Y'(m,x)*J'(m,labmda*x)
    %
    %     k = BESSCROSSZERO(m, labmda, N, BC, tol) as above, with roots
    %     calculated to relative tolerance 'tol' (default eps)
    %
    %     N can be an array
    numvarargs = length(varargin);
    optargs = {'DD', eps}; %Defaults
    optargs(1:numvarargs) = varargin;
    [T,tol] = optargs{:};
    if numel(T) ~= 2
        error('besscrosszero:invalidArgument', 'BC must be a string of length 2')
    end
    if ~((T(1) == 'D' || T(1) == 'N') && (T(2) == 'D' || T(2) == 'N'))
        error('besscrosszero:invalidArgument', 'BC elements must be either ''D'' or ''N''.')
    end
    %Transform roots if necessary
    if l < 1
        if strcmp(T, 'DN')
            T = 'ND';
        elseif strcmp(T,'ND')
            T = 'DN';
        end
        k = zerobesscross(nu,1/l,N,T,tol)/l;
        return
    end
    if l == 1
        k = nan(size(N));
        return
    end
    i = (T(1) == 'N');
    j = (T(2) == 'N');
    delta = ((i==1).*(nu==0) - 1).*(j==1);
    l1 = l-1;
    mu = 4*nu^2;
    tp = theta(i,j,nu,nu,l);
    k = zeros(size(N));
    if nargout > 1
        itter = zeros(size(N));
    end
    if nargout > 2
        errs = zeros(size(N));
    end
    % Initial guess for roots after the tp using McMahon's expansion
    mcm = ((N+delta)*pi >= tp);
    switch T
        case 'DD'
            k(mcm) = N(mcm)*pi/(l1);
            p = (mu-1)/(8*l);
            % q = (mu-1)*(mu-25)*(l^3-1)/(6*(4*l)^3*l1);
            % r = (mu-1)*(mu^2-114*mu+1073)*(l^5-1)/(5*(4*l)^5*l1);
        case 'DN'
            k(mcm) = (N(mcm)-0.5)*pi/(l1);
            p =-((mu+3)-(mu-1)*l)/(8*l*l1);
            % q =-((mu^2+46*mu-63)-(mu-1)*(mu-25)*l^3)./(6*(4*l)^3*l1);
            % r =-((mu^3+185*mu^2-2053*mu+1899) - (mu-1)*(mu^2-114*mu+1073)*l^5)/(5*(4*l)^5*l1);
        case 'ND'
            k(mcm) = (N(mcm)-0.5)*pi/(l1);
            p = ((mu+3)*l-(mu-1))/(8*l*l1);
            % q = ((mu^2+46*mu-63)*l^3-(mu-1)*(mu-25))./(6*(4*l)^3*l1);
            % r = ((mu^3+185*mu^2-2053*mu+1899)*l^5 - (mu-1)*(mu^2-114*mu+1073))/(5*(4*l)^5*l1);
        case 'NN'
            k(mcm) = (N(mcm)-1+(nu==0))*pi/(l1);
            p = (mu+3)/(8*l);
            % q = (mu^2+46*mu-63)*(l^3-1)./(6*(4*l)^3*l1);
            % r = (mu^3+185*mu^2-2053*mu+1899)*(l^5-1)/(5*(4*l)^5*l1);
    end
    if nu > 1
        k(mcm) = k(mcm) + p./k(mcm);
    end
    % For roots less than the order, use the roots/stationary points of the J_nu(l*x)
    if i == 0
        k(~mcm) = cylindzero(nu, N(~mcm), 0, 1e-3)/l;
    else
        k(~mcm) = cylindprimezero(nu, N(~mcm), 0, 1e-3)/l;
    end
    for m = 1:length(N)
        n = N(m);
        x = k(m);
        err = 1;
        itt = 0;
        while err > tol && itt < 25
            %Newton's method
            [t,xdt] = theta(i,j,nu,x,l);
            update = (t - (n+delta)*pi)./(xdt);
            x = x*(1-update);
            err = abs(update);
            itt=itt+1;
        end
        if nargout > 1
            itter(m) = itt;
        end
        if nargout > 2
            errs(m) = err;
        end
        if itt >= 25
            warning('unable to reach tol at, nu = %f, n = %d (last rel update %g, tol = %g)',nu,n,err,tol)
        end
        k(m) = x;
    end
end
function [t,xdt] = theta(i,j,n,x,l)
    if i == 1
        [t1,M2] = besselprimephase(n,l*x);
        xdt1 = 2*(1-(n./(l*x)).^2)./(pi*M2);
    else
        [t1,M2] = besselphase(n,l*x);
        xdt1 = 2./(pi*M2);
    end
    if j == 1
        [t2,M2] = besselprimephase(n,x);
        xdt2 = 2*(1-(n./x).^2)./(pi*M2);
    else
        [t2,M2] = besselphase(n,x);
        xdt2 = 2./(pi*M2);
    end
    t = t1-t2;
    xdt = xdt1 - xdt2;
end

function [x, itter] = cylindzero(nu, m, t, varargin)
    % x = CYLINDZERO(nu, m, t)  The mth positive real root of the function
    %   J(nu,x) cos(pi t) + Y(nu,x) sin(pi t) == 0
    % where J and Y are the Bessel functions of the first and second find.
    % 
    % m can be an array. nu and t must be scalars.
    %
    % The order nu must be postive.
    numvarargs = length(varargin);
    optargs = {2*eps}; %Defaults
    optargs(1:numvarargs) = varargin;
    [tol] = optargs{:};
    
    
    if length(nu) > 1
        error('cylindzero: the order must be a scalar')
    end
    if nu < 0
        error('cylindzero: the order must nonnegative')
    end
    if nargin < 3
        t = 0; 
    end
    % Which multiple of pi the first zero occurs at
    offset = -2 + (t <= 0);
    
    x = zeros(size(m));
    itter = zeros(size(m));
    for i=1:numel(m)
        targetPhase = (m(i)+offset+t+1/2)*pi;
        if m(i) == 1 && t > 0 && t <= 0.25
            if t < 1e-4
                warning('cylindzero: tolerance first root maybe be less than expected')
            end
            f =  @(x) besselphase(nu,x) - targetPhase;
            [x(i), n]  = ridders(f, 0, nu + 2*nu^(1/3) + 0.25, 10*eps);
            itter(i) = n;
            continue
        end
        % First term in McMahon's expansion for root
        x(i) = targetPhase + (2*nu + 1)*pi/4;
        relerr = 1;
        n = 0;
        while abs(relerr) > tol && n < 20
            [th,M2] = besselphase(nu,x(i));
            % From Newton's method
            relerr = (th-targetPhase)*pi*M2/2;
            x(i) = x(i)*(1 - relerr); 
            n = n + 1;
        end
        if n == 20
            error('cylindzero: failed to converge')
        end
        itter(i) = n;
    end
end



function [phase, modulus2] = besselphase(nu, x)
    % BESSELPHASE Phase function for the Bessel function derivatives
    % This private function does not contain argument checks. Please use
    % production code found in specphase package. 
    
    J = besselj(nu,x);
    Y = bessely(nu,x);
    % Fix matlab bug which results in incorrect overflow  near 0
    Y(isinf(Y)) = -Inf;
    phase = atan2(Y, J);
    if x > nu
        approx = sqrt(x^2 - nu^2)-nu*asec(x/nu) - pi/4;
        phase = phase - round((phase-approx)/(2*pi))*2*pi;
    end
    if nargout > 1
        modulus2 = (J.^2+Y.^2); 
    end
end


function [phase, modulus2] =  besselprimephase(nu,x)
    % BESSELPRIMEPHASE Phase function for the Besse function derivatives
    % This private function does not contain argument checks. Please use
    % production code found in specphase package. 
   
    dJ = 0.5*(besselj(nu-1,x)-besselj(nu+1,x));
    dY = 0.5*(bessely(nu-1,x)-bessely(nu+1,x));
    % Fix matlab bug which results in incorrect overflow 
    % and problems with our evaluation
    dY(isnan(dY) || isinf(dY)) = Inf;
    phase = atan2(dY,dJ);
    if x > nu
        approx = sqrt(x^2 - nu^2)-nu*asec(x/nu) + pi/4;
        phase = phase - round((phase-approx)/(2*pi))*2*pi;
    end
    if nargout > 1
        modulus2 = (dJ.^2+dY.^2); 
    end
end


function x = cylindprimezero(nu, m, t, varargin)
    % x = CYLINDPRIMEZERO(nu, m, t)  The mth positive real root of the function
    %     J'(nu,x) cos(pi t) + Y'(nu,x) sin(pi t) == 0 
    % where J' and Y' are the derivatives of Bessel functions of the first and
    % second find.
    % 
    % m can be an array. nu and t must be scalars.
    %
    % The order nu must be postive.
    numvarargs = length(varargin);
    optargs = {2*eps}; %Defaults
    optargs(1:numvarargs) = varargin;
    [tol] = optargs{:};
    
    
    if length(nu) > 1
        error('cylindprimezero: the order must be a scalar')
    end
    if nu < 0
        error('cylindprimezero: the order must nonnegative')
    end
    if nargin < 3
        t = 0; 
    end
    % Which multiple of pi the first zero occurs at
    offset = 0;
    
    [phaseNu, Modulus2Nu] = besselprimephase(nu,nu);
    phaseTurningPoint = phaseNu/pi - 1/2;
    %Root less than nu
    
    if t < 0
        offset = -1;
        if t == phaseTurningPoint %Double root
            offset = 0;
        end
        if t < phaseTurningPoint
            offset = 1;
        end
    end 
    
    if nu == 0 && t == 0
        offset = 1;
    end
    
    x = zeros(size(m));
    for i=1:numel(m)
        targetPhase = (m(i)+offset+t-1/2)*pi;
        if m(i) == 1 
            if t == phaseTurningPoint && nu ~= 0
                x(i) = nu;
                continue
            end
            if offset == -1
                f =  @(x) besselprimephase(nu,x) - (t + 1/2)*pi;
                x(i)  = ridders(f, 0, nu, 2*eps);
                continue
            end
        end
        % First term in McMahon's expansion for root
        x(i) = targetPhase + (2*nu - 1)*pi/4;
        if m(i) == 2 && offset == -1
            x(i) = nu*(1 + pi*sqrt(Modulus2Nu/2*(t - phaseNu/pi+1/2)));
        end
        relerr = 1;
        n = 0;
        while abs(relerr) > tol && n < 20
            [th,M2] = besselprimephase(nu,x(i));
            % Newton's method
            relerr = (th-targetPhase)*pi/2*M2/(1-(nu/x(i))^2);
            x(i) = x(i)*(1 - relerr);
            n = n + 1;
        end
        if n == 20
            warning(['cylindprimezero: Failed to attain tolerence ' ... 
                     'on root m(%d) = %d. Instead root has' ...
                     'relative error = %5g.'], ...
                     i, m(i), relerr)
        end
    end
end
