function geom = cubicplanarbend3D(geomparams)


%% PARAMETERS
%duct start width R_i, length s_o, decrease ratio > 1, curvature, bend
%radius R_B and bend angle theta_B
if ~exist('geomparams','var')
    geomparams.R_i      = 0.5;
    geomparams.s_o      = 3.275*2*geomparams.R_i;
    geomparams.ratio    = 4;
    geomparams.kappa    = 4/(5*2*geomparams.R_i);
end

R_i     = geomparams.R_i;
s_o     = geomparams.s_o;
ratio   = geomparams.ratio;
kappa   = geomparams.kappa;
R_B     = 1/kappa;
theta_B = s_o/R_B;

geom.geomparams = geomparams;

%% SCALAR S-FUNCTIONS
    function R = ductradius(s)
        if s < s_o
            R = R_i*(1 - 1/ratio)*(s/s_o)^2*(2*s/s_o - 3) + R_i;
        else
            R = R_i/ratio;
        end
    end
    function Rdash = ductradiusderiv(s)
        if s <= s_o
            Rdash = (6*R_i)*(1 - 1/ratio)*(s/s_o)*(s/s_o - 1);
        else
            Rdash = 0;
        end
    end
geom.L      = s_o;
geom.R      = @(s) ductradius(s);
geom.Rdash  = @(s) ductradiusderiv(s);
geom.kappa  = @(s) kappa;
geom.tau    = @(s) 0;
geom.theta0 = @(s) 0;
geom.qyvar  = true;
geom.qzvar  = false;
geom.dim    = 3;


%% VECTOR S-FUNCTIONS
    function x = ductpathx(s,parity)
        x = parity*sin(kappa*s);
    end
    function y = ductpathy(s,parity,cl)
        y = (cl*R_B - parity*cos(kappa*s))...
            *sign(theta_B);
    end
    function x = innerductpathx(s,parity)
        x = -parity*sin(abs(theta_B) - kappa*s);
    end
    function y = innerductpathy(s,parity,cl)
        y = (cl*R_B - parity*cos(abs(theta_B) - kappa*s));
    end

geom.qx = @(s) ductpathx(s,R_B);
geom.qy = @(s) ductpathy(s,R_B,1);
geom.qz = @(s) 0;
geom.nx = @(s) ductpathx(s,-1);
geom.ny = @(s) ductpathy(s,-1,0);
geom.nz = @(s) 0;
geom.bx = @(s) 0;
geom.by = @(s) 0;
geom.bz = @(s) sign(theta_B);

geom.innerqx = @(s) innerductpathx(s,R_B);
geom.innerqy = @(s) innerductpathy(s,R_B,1);
geom.innernx = @(s) innerductpathx(s,-1);
geom.innerny = @(s) innerductpathy(s,-1,0);



%% GEOMETRY NAME
geom.name = 'cubicplanarbend3D';

end