clc
clear 
set(0,'DefaultFigureWindowStyle','docked')

%%% Calling a function needed for creating colormap
Ncols = 256;
colmap = zeros(Ncols,3);
for col = 1:Ncols
    colmap(col,:) = custom_color2((col-1)/(Ncols-1));
end

%% Initial Parameters as inputs

tensionIn_N = 0; % Tension at the roll gap entrance in N per unit width
tensionOut_N = 0; % Tension at the roll gap exit in N per unit width
shearYieldStress_P = 5e8/sqrt(3);

initalworkpieceThickness_m= 0.002*2;
reductionRatio= 0.25; 
% rollerRadius_m = 100*(10^-3); 
% frictionCoefficient= 0.1; 
 
% delta=0.075; frictionCoefficient= 0.1;  
 delta=0.125; frictionCoefficient= 0.1;  
% delta=0.175; frictionCoefficient= 0.1;  
% delta=0.2; frictionCoefficient= 0.1; 
% delta=0.3; frictionCoefficient= 0.15; 
% delta=0.5; frictionCoefficient= 0.25;  

vertialElements= 500; % Number of discritisations in the vertical direction
horizontalElements= 3*vertialElements; % Number of discretisiations in the horizontal direction

%% definig needed parameters
tic

wr = initalworkpieceThickness_m/2*reductionRatio; % This parameter is equal to(t1-t2)/2
wt = initalworkpieceThickness_m/2; % Workpiece half-thickness in meter = t1/2
rt = wr/wt;  % Non-dimensioned wr
%R = rollerRadius_m;
R = wt * ((1./(delta.^2))+ reductionRatio^2) / (2*reductionRatio); % Roll radius
wl = sqrt(2*R*wr - wr^2); % Length of roll-gap in m %sqrt(rR^2 - (rR-wr)^2)
delta = wt/wl;  % Ratio of half thickness to the length of roll gap

mu = frictionCoefficient;
beta = mu / delta; % Scaled friction coefficient with delta

k = shearYieldStress_P;
Fin = tensionIn_N; % Force at the roll gap entrance in N
Fout = tensionOut_N; % Force at the roll gap exit in N                  
F_in = (Fin)/(k*wt*4); % Force per unit width per unit length scaled by k
F_out = (Fout)/(k*wt*4);

nz = horizontalElements+1; % Number of nodes in the horizontal direction
ny = vertialElements+1; % Number of nodes in the vertical direction
z = linspace(0,1,nz);
y = linspace(-1,1,ny);
[Z,Y] = meshgrid(z, linspace(1,0,ny));

h = function_h(z, delta, rt);
hprime = function_hprime(z, delta, rt);

for i=1:nz
Y(:,i)=linspace(-h(i),h(i),ny);
end

%% leading pressure

% Initial condition for leading-order pressure at the entrance and exit
p_initial = [ 1-F_in , 1-F_out ]; 

% Solving leading-order pressure from entrance onwards
[z_left,p_left] = ode45(@(zt,pt) p_ode_left(zt,pt, beta, delta, rt), z, p_initial(1)); 

% Solving leading-order pressure from exit backwards
[z_right,p_right] = ode45(@(zt,pt) p_ode_right(zt,pt, beta, delta, rt), fliplr(z), p_initial(2)); 

% Find the intersection of the two above solution in z axis
p_left_int = @(zt)interp1(z_left, p_left, zt); p_left = p_left_int(z);
p_right_int = @(zt)interp1(z_right, p_right, zt); p_right = p_right_int(z);
zN = fminbnd(@(z)abs((p_left_int(z))-(p_right_int(z))), 0.0, 1.0); % Location of the neutral point in z axis

%  Find the location and value of the nearst point to the neutral point in  z array
[val1,idzN] = min(abs(z(1,:)-zN)); 
zN=z(1,idzN);

% Definig leading-order pressure (p^{(0)})
p = p_left .* (z<zN) + p_right .* (z>=zN); 

% Definig dp^{(0)}/dz 
p_prime_left = p_ode_left(z,p_left, beta, delta, rt);
p_prime_right = p_ode_right(z,p_right, beta, delta, rt);
p_prime = p_prime_left .* (z<zN) + p_prime_right .* (z>=zN);
p_prime_int = @(zt) interp1(z, p_prime, zt);


%% Definig small-scale parameter n:
    
n = cumtrapz (z, 1 ./ (delta * h)); % n = int dz / (h(z) * delta)
nN = n(idzN); % value of n at the neutral ponit

%% parameters needed for Burger's equations

log_alpha1_left = 0.5 * beta *  cumtrapz (z(1:idzN), (1 - p_left(1:idzN)) ./ h(1:idzN));
log_alpha1_right = log_alpha1_left(end) - 0.5 * beta *  cumtrapz (z(idzN+1:end), (1 - p_right(idzN+1:end)) ./ h(idzN+1:end));
log_alpha1= [log_alpha1_left,log_alpha1_right];
alpha1 = exp(log_alpha1); 

T = cumtrapz(z, alpha1 ./ h.^2); 

%% Calculating A(ξ,z) for the first-order correction

% Defining ξ axis
nxi = 2*ny+1; % number of grids in ξ domain. To increase accuracy the grid is made finner 
xi = linspace(-1,1,nxi); % ξ domain
dxi = xi(2) - xi(1);

% Definig the initial condition for A(ξ,z) at the entrance 
a1 = -(0) ; % Quadratic factor
a2 = -(1.2 + p_prime(1)) / 2; % Linear factor
A0 =  (( 3*a1*(xi+1).^2 + a2*(xi+1) - a1).*(xi<0) + ( 3*a1*(xi-1).^2  + a2*(xi-1) - a1).*(xi>=0)); % ξ=-1 is located at z=0, and xi = 1 is located at 2  

% Definig initial condition for w(ξ,T) from w = A * h/ alpha1
w0_left   = A0 * h(1) / alpha1(1); 

% Solving Burger's equation from entrance up to the neutral point
[T_left,w_left] = ode45 (@Burgers , T(1:idzN) , w0_left);

% Finding the location of the neutral point in ξ between [-1,1]
xiN = (nN + 1) - 2 *(ceil(nN/2));
[val2,idxiN] = min(abs(xi(1,:)-xiN)); % array number of the neutral point in idxi between [1,2*ny+1]   
xiN_end = -xiN;
[val3,idxiN_end] = min(abs(xi(1,:)-xiN_end)); 

% Definig the new wave initiating from the neutral point located in ξ between [-1,1]
extra = ( (beta*(p(idzN)+1)) * ( (xi+1).*(xi<=0) +  (xi-1).*(xi>0) )) * h(idzN) / alpha1(idzN); 
extra = [extra(idxiN_end:end),extra(1:idxiN_end-1)];

% Definig the initial condition for w(ξ,z) at the neutral point
w0_right = w_left(end,:) + extra ; %%% w0_right = w_left(zN) + extra

% Solving Burger's equation from the neutral point up to the exit
[T_right,w_right] = ode45 (@Burgers , T(idzN:end) , w0_right);

% Definig w from the entrance to the exit
w = [w_left(1:end-1,:) ; w_right]; % ξ is horizonatal coordinates and T is vertical coordinate
w = w'; %ξ is vertiacl coordinate (top is -1 and bottom is 1) and T (or z) is horizoantal coordinate

% Defining A(ξ,z) from the entrance to the exit
A = w .* alpha1 ./ h; % w = A * h / alpha 
                      % ξ is vertiacl coordinate (top is -1 and bottom is 1) and T (or z) is horizoantal coordinate

% Defining (n + y/h) and (n - y/h) such that it is always between -1 and 1
xiplus = (n + Y./h - 1 ) - 2 *(floor((n + Y./h)/2)); 
ximinus = (n - Y./h - 1 ) - 2 *(floor((n - Y./h )/2));

% Defining A(n + y/h,z) and A(n - y/h,z)
[z_mat,xi_mat]=meshgrid(z,xi); % z is 1*nz vector. xi is  2ny*1 vector 
Axipls = interp2(z_mat,xi_mat, A, Z, xiplus,'cubic');  % A for ξ=n+y/h (between -1 and 1)
Aximin = interp2(z_mat,xi_mat, A, Z, ximinus,'cubic');  % A for ξ=n-y/h (between -1 and 1)


%% Definig Pressure, Shear, sigma_xx, horizontal and vertical velocities

% Pressure 
press_LO = -(p + 0*Y); 
press_FO = -(Axipls + Aximin) ;
press = ( press_LO + delta*press_FO);

% Sigma_xx
sigma_xx_LO = 1 + press_LO;
sigma_xx = ( sigma_xx_LO  + delta*press_FO);

% Sigma_yy
sigma_yy_LO = -1 + press_LO;
sigma_yy = (sigma_yy_LO + delta*press_FO);

% Shear
shear_LO = delta * (p_prime .*Y);
shear = ( shear_LO + delta*(Axipls - Aximin) );
shear_SO = delta^2 *(- beta* press_FO(end,:) .* (z<zN) + beta* press_FO(end,:) .* (z>=zN)); % Second order correction for shear (only valid on surface)

%% Time needed to compute all the variable (without plotting)
fprintf('Calculation time: %.2f seconds. \n',toc')

%% Plotting Pressure hill:
fprintf(1,"Plotting pressure hill... ");
figure('Name','hill pressure','NumberTitle','off')

plot (z,p_left,z,p_right,'LineWidth',3)
grid on

title('$$\mathbf{p^{(0)}}$$','Interpreter','latex','FontSize',36)
xlabel('$$\mathbf{z}$$','Interpreter','latex')
ylabel('$$\mathbf{y}$$','Interpreter','latex')
lg = legend('left','right');


set(gcf,'color','w'); %%% change the background color to white
set(gca,'LineWidth',3, 'FontSize',36) %%% change the width of border of graph and the font size of all axes (numbers, labels, colorbar)

pbaspect([2.2 1 1])

fprintf(1,"done.\n");

%% Plotting shear
fprintf(1,"Plotting sigma_xy... ");
figure('Name','Shear','NumberTitle','off')

color_max = max(max([abs(shear_LO), abs(shear)]));
color_min = -color_max;

contourf(Z, Y , shear ,14,'LineWidth',3);

clim ([color_min,color_max])
colormap(colmap);
colorbar
hcb=colorbar;

title('$$\mathbf{\sigma_{xy}}$$','Interpreter','latex','FontSize',36)
xlabel('$$\mathbf{z}$$','Interpreter','latex')
ylabel('$$\mathbf{y}$$','Interpreter','latex')

set(gcf,'color','w'); %%% change the background color to white
set(gca,'LineWidth',3, 'FontSize',36) %%% change the width of border of graph and the font size of all axes (numbers, labels, colorbar)

pbaspect([2 1 1])

fprintf(1,"done.\n");
%% Plotting Pressure
fprintf(1,"Plotting pressure... ");
figure('Name','pressure','NumberTitle','off')

color_max = max(max([abs(press_LO), abs(press)]));
color_min = -color_max;

contourf(Z, Y , press ,22,'LineWidth',3)


color_max=0;
clim ([color_min,color_max])
colormap(colmap);
%colormap(colmap(1:col/2,:)); 
% colormap(brewermap([],"RdBu"))
colorbar

title('$$\mathbf{{p}}$$','Interpreter','latex','FontSize',36)
xlabel('$$\mathbf{z}$$','Interpreter','latex')
ylabel('$$\mathbf{y}$$','Interpreter','latex')
set(gcf,'color','w'); %%% change the background color to white
set(gca,'LineWidth',3, 'FontSize',36) %%% change the width of border of graph and the font size of all axes (numbers, labels, colorbar)

pbaspect([2 1 1])

fprintf(1,"done.\n");

%% Plotting Sigma_xx
fprintf(1,"Plotting Sigma_xx... ");
figure('Name','Sigma_{xx}}','NumberTitle','off')

color_max = max(max([abs(sigma_xx_LO), abs(sigma_xx)]));
color_min = -color_max;

contourf(Z, Y ,  sigma_xx ,22,'LineWidth',3)


clim ([color_min,color_max])
colormap(colmap);
colorbar 

title('$$\mathbf{\sigma_{xx}}$$','Interpreter','latex','FontSize',36)
xlabel('$$\mathbf{z}$$','Interpreter','latex')
ylabel('$$\mathbf{y}$$','Interpreter','latex')

set(gcf,'color','w'); %%% change the background color to white
set(gca,'LineWidth',3, 'FontSize',36) %%% change the width of border of graph and the font size of all axes (numbers, labels, colorbar)

pbaspect([2 1 1])

fprintf(1,"done.\n");

%% Functions:

function h=function_h(zt, delta, rt)
% This function calculate h(z)

R = (1/delta^2+rt^2)/(2*rt);
h=(1-rt)+R - sqrt(R^2 - ((zt-1)/delta).^2);
end

function hprime = function_hprime(zt, delta, rt)
% This function calculate dh(z)/dz
R = (1/delta^2+rt^2)/(2*rt);
hprime = (zt-1)./(sqrt(R^2 - ((zt-1)/delta).^2)*delta^2);
end

function dpdz_left = p_ode_left(zt,pt, beta, delta, rt)
% This function is used for solving pressure hill from the entrace forwards

h = function_h(zt, delta, rt);
hprime = function_hprime(zt, delta, rt);
dpdz_left =  (2*hprime + beta*(1 + pt)) ./ h;
end

function dpdz_right = p_ode_right(zt,pt, beta, delta, rt)
% This function is used for solving pressure hill from the exit backwards

h = function_h(zt, delta, rt);
hprime = function_hprime(zt, delta, rt);
dpdz_right = (2*hprime - beta*(1 + pt)) ./ h;
end

function dwdT = Burgers(~,w)
% This function is used for solving Burger's equation

mm = length(w);
dxi = 2 /(mm-1);

w = [w(end);w;w(1)];
fc = 0.5 * w.^2;
fb = 0.5 * (fc(1:end-1) + fc(2:end));
a  = max(abs(w(1:end-1)), abs(w(2:end)));
d = w(1:end-1) - w(2:end);
L = 2 * d(1:end-2) .* abs(d(3:end)) ./ (abs(d(1:end-2)) + abs(d(3:end)));
L(d(1:end-2) .* d(3:end) <= 0) = 0;
d(2:end-1) = d(2:end-1) - L;
fd = 0.5 * a .* d;
fb = fb + fd;
if w(1) < 0 
    fb(1) = fc(1); 
end
if w(end) > 0 
    fb(end) = fc(end); 
end
dwdT = (fb(1:end-1) - fb(2:end)) / dxi;
    
end


function triple = custom_color2(gray)
% This function creates nice colors for plotting

    gray = 2*gray;
    if (gray < 1)
        r=gray;
        g=gray;
        b=1;
    else
        r=1;
        g=2-gray;
        b=2-gray;
    end
    triple = [r,g,b];
end

