function sandwichRoll()
%%% Uses asymptotic methods to calculate the details of sandwich rolling.
%%% Assumptions: 
%%%   * Rigid-perfectly-plastic material
%%%   * No roll deformation
%%%   * Coulomb friction with constant coefficient
%%%   * Small roll gap aspect ratio
%%%
%%% To use, set physical parameters in the block below.
%%% Edit functions hfunc, dhdxfunc etc to specify roller shape if desired.
%%% 
%%% @author:  Chris Cawthorn (c.j.cawthorn@damtp.cam.ac.uk)
%%% @version: v1.0 (16 June 2015)  
%%% 
%%% Supplementary material to 
%%% "Asymptotic analysis of cold sandwich rolling"
%%% by Cawthorn, Minton & Brambley.
%%% Submitted to Int. J. Mech. Sci in June 2015. 

%%% PHYSICAL PARAMETERS %%%
rR = 100;         % Roll radius in mm (moderate 100, large 400)
ww = 100;         % Strip width in mm
h0 = 5;           % Initial strip half-thickness in mm
dh = 1;           % Strip half-reduction in mm
alpha = 0.7;      % Relative thickness of matrix at entrance
kM = 300/sqrt(3);         % Yield stress of matrix layer, in MPa
kC = 300/sqrt(3)*0.8;     % Yield stress of clad layer, in MPa
mu = 0.1;         % Coulomb friction coefficient between clad & roller
angvel = 0.1;     % Angular velocity in rad/s
TinD = 0;         % Tension at roll gap entrance, in N
ToutD = 0;        % Tension at roll gap exit, in N
%%% NUMERICAL PARAMETERS %%%
nx = 201;
ny = 501;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Make a call to compute a single result.
% The sandwichResult struct contains the following fields:
% (*0 is leading order, *corr has the 2nd order correction)
% x,y                     : Positions in mm 
% u0,ucorr,v0,vcorr       : Velocity in mm/s
% sigmaxx0,sigmaxxcorr    : Cauchy xx-stress in MPa
% sigmayy0,sigmayycorr    : Cauchy yy-stress in MPa
% Sxy0,Sxycorr            : Cauchy xy-stress in MPa
% xN                      : Neutral point position from gap entrance in mm
% force0,forcecorr        : Roll force in N
% torque0,torquecorr      : Roll torque in Nm
%
result = singleCalculation(rR,ww,h0,dh,alpha,kM,kC,mu,angvel,TinD,ToutD,nx,ny);
display(['Roll force:     ',num2str(result.forcecorr/1000),' kN'])
display(['Roll torque:    ',num2str(result.torquecorr/1000),' kNm'])

end

        
function result = singleCalculation(rR,ww,h0,dh,alpha,kM,kC,mu,TinD,ToutD,angvel,nx,ny)
%%% Computational engine for the sandwich roll calculation
%%% Returns a struct with fields for force, torque, velocity & stress
%%%

%%% DERIVED QUANTITIES %%%
beta = kC/kM;              % Ratio of yield stresses
ell  = sqrt(2*rR*dh);      % Horizontal lengthscale (parabolic roller)
delta = h0/ell;            % Aspect ratio
red   = dh/h0;             % Dimensionless reduction
muOD  = mu/delta;          % "Order 1" mu, i.e. mu/delta. 
sigmaxxin  = TinD/2/h0/ww/kM;       % Dimensionless horizontal stess at entrance
sigmaxxout = ToutD/2/(h0-dh)/ww/kM; % Dimensionless horizontal stress at exit  
%%%%%%%%%%%%%%%%%%%%%%%%%%

% Set up solution grid
xv = linspace(0,1,nx);
yv = linspace(0,1,ny);
[x,y] = meshgrid(xv,yv);
hv      = hfunc(xv,red,rR,ell,h0);
h       = hfunc(x,red,rR,ell,h0);
dhdxv   = dhdxfunc(xv,red,rR,ell,h0);
dhdx    = dhdxfunc(x,red,rR,ell,h0);
d2hdx2v = d2hdx2func(xv,red,rR,ell,h0);
d2hdx2  = d2hdx2func(x,red,rR,ell,h0);
d3hdx3v = d3hdx3func(xv,red,rR,ell,h0);
d3hdx3  = d3hdx3func(x,red,rR,ell,h0);

%% First order
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% LEADING ORDER SOLUTION %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Find the leading order solution
% We want the mean horizontal stress to be sigmaxxin
% but sigmaxx = { sigma0  + 2      in matrix
%               { sigma0  + 2*beta in clad
% Thus, mean stress is (sigma0 + 2(beta*(1-alpha) + alpha)
sigma0in0  = sigmaxxin - 2*(beta*(1-alpha) + alpha);
% The same applies at the exit, noting that the thickness ratio is constant
% at leading order
sigma0out0 = sigmaxxout - 2*(beta*(1-alpha) + alpha);
% Solve the ODE in the entrance
[xsolin, sig0solin] = ode45(@(x,sigma0) odefun0in(x,sigma0,alpha,beta,muOD,red,rR,ell,h0),[0,1],sigma0in0);
% Solve the ODE in the exit
[xsolout, sig0solout] = ode45(@(x,sigma0) odefun0out(x,sigma0,alpha,beta,muOD,red,rR,ell,h0),[1,0],sigma0out0);
% Define xN to be the position at which these two coincide
xN = fsolve(@(x) xNfunc(x,xsolin,sig0solin,xsolout,sig0solout),0.5);
hN = interp1(xv,hv,xN);

% Find the combined sigma0*(=sigmayy)
sigmayy0inv  = interp1(xsolin,sig0solin,xv);
sigmayy0outv = interp1(xsolout,sig0solout,xv);
sigmayy0v = (xv<=xN).*sigmayy0inv + (xv>xN).*sigmayy0outv;
sigmayy0 = kron(sigmayy0v,ones(ny,1));

% Find sigmaxx in each layer, and amalgamate
sigmaxx0M = sigmayy0 + 2;
sigmaxx0C = sigmayy0 + 2*beta;
sigmaxx0 = (y<=alpha*h).*sigmaxx0M + (y>alpha*h).*sigmaxx0C;

% Compute the derivatives of sigma0in and sigma0out 
% (analytically from sigma0)
% First derivative
dsigma0inv  = 2*(alpha*beta-beta-alpha)*dhdxv./hv + muOD*sigmayy0inv./hv;
dsigma0outv = 2*(alpha*beta-beta-alpha)*dhdxv./hv - muOD*sigmayy0outv./hv;
dsigma0in  = kron(dsigma0inv,ones(ny,1));
dsigma0out = kron(dsigma0outv,ones(ny,1)); 
% Second derivative
ddsigma0inv  = (2*(alpha*beta-beta-alpha)*d2hdx2v - dhdxv.*dsigma0inv + muOD*dsigma0inv)./hv;
ddsigma0outv = (2*(alpha*beta-beta-alpha)*d2hdx2v - dhdxv.*dsigma0outv - muOD*dsigma0outv)./hv;
ddsigma0in = kron(ddsigma0inv,ones(ny,1));
ddsigma0out = kron(ddsigma0outv,ones(ny,1)); 
% Third derivative
dddsigma0inv  = (2*(alpha*beta-beta-alpha)*d3hdx3v - 2*dhdxv.*ddsigma0inv  - d2hdx2v.*dsigma0inv  + muOD*ddsigma0inv)./hv;
dddsigma0outv = (2*(alpha*beta-beta-alpha)*d3hdx3v - 2*dhdxv.*ddsigma0outv - d2hdx2v.*dsigma0outv - muOD*ddsigma0outv)./hv;
dddsigma0in = kron(dddsigma0inv,ones(ny,1));
dddsigma0out = kron(dddsigma0outv,ones(ny,1)); 

% Compute Sxy0 in each region (SxyinC, SxyoutC, SxyinM, SxyoutM)
Sxy0Min  = -y.*dsigma0in/muOD;
Sxy0Mout = -y.*dsigma0out/muOD;
Sxy0M    = (x<=xN).*Sxy0Min + (x>xN).*Sxy0Mout;
Sxy0Cin  = -y.*dsigma0in/muOD  + 2*(beta-1)*alpha*dhdx/muOD;
Sxy0Cout = -y.*dsigma0out/muOD + 2*(beta-1)*alpha*dhdx/muOD;
Sxy0C    = (x<=xN).*Sxy0Cin + (x>xN).*Sxy0Cout;
Sxy0     = (y<=alpha*h).*Sxy0M + (y>alpha*h).*Sxy0C;

% Compute the leading order plastic multiplier
% These are the same in the clad and matrix regions
lambda0Mv = -hN*dhdxv./hv.^2;
lambda0M  = -hN*dhdx./h.^2;
lambda0Cv = -hN*dhdxv./hv.^2/beta;
lambda0C  = -hN*dhdx./h.^2/beta;
lambda0   = (y<=alpha*h).*lambda0M + (y>alpha*h).*lambda0C;

% Compute the leading order velocity
u0v = hN./hv; 
u0  = hN./h;
v0  = hN*y.*dhdx./h.^2;   % v0 is y-dependent. No vector needed.


%% Second order
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% SECOND ORDER SOLUTION %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Find horizontal velocity
% x-dependent part u2m
u2minv  = hN*d2hdx2v/6 - hN*dhdxv.^2/3./hv - hN*dhdxv.*dsigma0inv/3/beta  + (beta-1)/beta*alpha*hN*(2*(alpha-1)^2*dhdxv.^2./hv+alpha*(2*alpha-3)*dhdxv.*dsigma0inv/3);
u2moutv = hN*d2hdx2v/6 - hN*dhdxv.^2/3./hv - hN*dhdxv.*dsigma0outv/3/beta + (beta-1)/beta*alpha*hN*(2*(alpha-1)^2*dhdxv.^2./hv+alpha*(2*alpha-3)*dhdxv.*dsigma0outv/3);
u2min   = kron(u2minv,ones(ny,1));
u2mout  = kron(u2moutv,ones(ny,1));
% Horizontal velocity in matrix region
uM2in  = u2min  + hN*y.^2./2./h.^3.*(2*dhdx.^2-h.*d2hdx2+2*h.*dhdx.*dsigma0in);
uM2out = u2mout + hN*y.^2./2./h.^3.*(2*dhdx.^2-h.*d2hdx2+2*h.*dhdx.*dsigma0out);
uM2    = (x<=xN).*uM2in + (x>xN).*uM2out;
% Horizontal velocity in clad region
uC2in  = u2min  + alpha^2*hN./h*(beta-1)/beta.*(4*dhdx.^2+h.*dhdx.*dsigma0in ) - 4*alpha*(beta-1)/beta*hN*dhdx.^2.*y./h.^2 + hN/2./h.^3.*(2*dhdx.^2-h.*d2hdx2+2*h.*dhdx.*dsigma0in /beta).*y.^2;
uC2out = u2mout + alpha^2*hN./h*(beta-1)/beta.*(4*dhdx.^2+h.*dhdx.*dsigma0out) - 4*alpha*(beta-1)/beta*hN*dhdx.^2.*y./h.^2 + hN/2./h.^3.*(2*dhdx.^2-h.*d2hdx2+2*h.*dhdx.*dsigma0out/beta).*y.^2;
uC2    = (x<=xN).*uC2in + (x>xN).*uC2out;

% Find vertical velocity 
% First find the derivative of u2m
du2minv  = hN*d3hdx3v/6 + hN*(2*alpha*(alpha-1)^2*(beta-1)/beta-1.0/3)*(2*dhdxv.*d2hdx2v./hv-dhdxv.^3./hv.^2) + hN/3/beta*((2*alpha-3)*alpha^2*(beta-1)-1)*(d2hdx2v.*dsigma0inv +dhdxv.*ddsigma0inv );
du2moutv = hN*d3hdx3v/6 + hN*(2*alpha*(alpha-1)^2*(beta-1)/beta-1.0/3)*(2*dhdxv.*d2hdx2v./hv-dhdxv.^3./hv.^2) + hN/3/beta*((2*alpha-3)*alpha^2*(beta-1)-1)*(d2hdx2v.*dsigma0outv+dhdxv.*ddsigma0outv);
du2min   = kron(du2minv,ones(ny,1));
du2mout  = kron(du2moutv,ones(ny,1));
% Vertical velocity in the matrix region
vM2in  = -y.*du2min  - hN*y.^3/6./h.^4.*(6*h.*dhdx.*d2hdx2 - 6*dhdx.^3 - h.^2.*d3hdx3 + 2*h.^2.*d2hdx2.*dsigma0in  + 2*h.^2.*dhdx.*ddsigma0in  - 4*h.*dhdx.^2.*dsigma0in);
vM2out = -y.*du2mout - hN*y.^3/6./h.^4.*(6*h.*dhdx.*d2hdx2 - 6*dhdx.^3 - h.^2.*d3hdx3 + 2*h.^2.*d2hdx2.*dsigma0out + 2*h.^2.*dhdx.*ddsigma0out - 4*h.*dhdx.^2.*dsigma0out);
vM2    = (x<=xN).*vM2in + (x>xN).*vM2out;
% Vertical velocity in the clad region 
vC2in  = -y.*du2min  + alpha^3*hN/3*(beta-1)/beta*(2*dhdx.^2.*dsigma0in  - h.*d2hdx2.*dsigma0in  - h.*dhdx.*ddsigma0in ) ...
     - alpha^2*hN*(beta-1)/beta.*(4*dhdx.^3./h.^2+d2hdx2.*dsigma0in  + dhdx.*ddsigma0in ).*(y-alpha*h) ...
     + 4*alpha*hN*(beta-1)/beta./h.^3.*(h.*dhdx.*d2hdx2 - dhdx.^3).*(y-alpha*h).^2 ...
     + hN*y.^3/6./h.^3.*(6*dhdx.^3./h + h.*d3hdx3 - 6*dhdx.*d2hdx2 + 4*dhdx.^2.*dsigma0in /beta - 2*h.*d2hdx2.*dsigma0in /beta  -2*h.*dhdx.*ddsigma0in /beta);
vC2out = -y.*du2mout + alpha^3*hN/3*(beta-1)/beta*(2*dhdx.^2.*dsigma0out - h.*d2hdx2.*dsigma0out - h.*dhdx.*ddsigma0out) ...
     - alpha^2*hN*(beta-1)/beta.*(4*dhdx.^3./h.^2+d2hdx2.*dsigma0out + dhdx.*ddsigma0out).*(y-alpha*h) ...
     + 4*alpha*hN*(beta-1)/beta./h.^3.*(h.*dhdx.*d2hdx2 - dhdx.^3).*(y-alpha*h).^2 ...
     + hN*y.^3/6./h.^3.*(6*dhdx.^3./h + h.*d3hdx3 - 6*dhdx.*d2hdx2 + 4*dhdx.^2.*dsigma0out/beta - 2*h.*d2hdx2.*dsigma0out/beta  -2*h.*dhdx.*ddsigma0out/beta);
vC2    = (x<=xN).*vC2in + (x>xN).*vC2out; 

% Compute the correction to xN
uC2hin  = -(d2hdx2v-2*dhdxv-2*hv.*dhdxv.*dsigma0inv/beta)/3  + alpha*(beta-1)/beta*(2*(alpha^2-1)*dhdxv.^2+2*alpha*hv.*dhdxv.*dsigma0inv/3);
uC2hout = -(d2hdx2v-2*dhdxv-2*hv.*dhdxv.*dsigma0outv/beta)/3 + alpha*(beta-1)/beta*(2*(alpha^2-1)*dhdxv.^2+2*alpha*hv.*dhdxv.*dsigma0outv/3);
% Evaluate each at xN and take the mean
uC2xNin = interp1(xv,uC2hin,xN);
uC2xNout = interp1(xv,uC2hout,xN);
uC2xN = 0.5*(uC2xNin+uC2xNout);
% Evaluate h' at xN
dhdxxN = interp1(xv,dhdxv,xN);
% Compute xN2
xN2 = hN*(0.5*dhdxxN^2-uC2xN)/dhdxxN;
%xN0 = xN
%xN = xN0 + delta^2*xN2
% Re-establish u and v given the updated xN
%uM2    = (x<=xN).*uM2in + (x>xN).*uM2out;
%uC2    = (x<=xN).*uC2in + (x>xN).*uC2out;
%vM2    = (x<=xN).*vM2in + (x>xN).*vM2out;
%vC2    = (x<=xN).*vC2in + (x>xN).*vC2out; 

% Compute correction to the interface
% Begin by finding v2 and u2 on at alpha*h
% Note: uC2 and uM2 should be the same here (likewise vC2 and vM2)
u2inonhm  = interp2(x,y,uM2in,xv,alpha*hv);
u2outonhm = interp2(x,y,uM2out,xv,alpha*hv);
v2inonhm  = interp2(x,y,vM2in,xv,alpha*hv);
v2outonhm = interp2(x,y,vM2out,xv,alpha*hv);
% Find dhm2/dx in entrance and exit region
% Actually, it's d(hm2/h)/dx. We'll multiply by h later.
dHM2dxinv  = (v2inonhm -alpha*dhdxv.*u2inonhm )/hN;
dHM2dxoutv = (v2outonhm-alpha*dhdxv.*u2outonhm)/hN;
% Integrate from zero at each end
[xsolin,hm2solin]   = ode45(@(xval,hm2val) interp1(xv,dHM2dxinv, xval),[0,1],0.0); 
[xsolout,hm2solout] = ode45(@(xval,hm2val) interp1(xv,dHM2dxoutv,xval),[1,0],0.0); 
% Interpolate back onto the solution mesh
hm2inv  = interp1(xsolin,hm2solin,xv);
hm2outv = interp1(xsolout,hm2solout,xv);
% Amalgamated hm2
% NOTE: This will almost certainly be discontinuous at xN. This is a
% consequence of the discontinuity in Sxy.
hm2v = (xv<=xN).*hm2inv + (xv>xN).*hm2outv;
hmv  = alpha*hv + delta^2*hm2v;
hm2  = kron(hm2v,ones(ny,1));
hm   = alpha*h  + delta^2*hm2;
% We should now re-amalgamate the velocities and stresses taking
% into account the modified interface position. In practice, it's almost
% indisitinguishable to the naked eye.
u2        = (y<=hm).*uM2       + (y>hm).*uC2;
v2        = (y<=hm).*vM2       + (y>hm).*vC2;
sigmaxx0  = (y<=hm).*sigmaxx0M + (y>hm).*sigmaxx0C;
Sxy0      = (y<=hm).*Sxy0M     + (y>hm).*Sxy0C;
lambda0   = (y<=hm).*lambda0M  + (y>hm).*lambda0C;
% Finally, find the derivative of hm2. We'll need this later.
dhm2dxinv  = 0*hv.*dHM2dxinv  + dhdxv.*hm2inv./hv;
dhm2dxoutv = 0*hv.*dHM2dxoutv + dhdxv.*hm2outv./hv;
dhm2dxin   = kron(dhm2dxinv,ones(ny,1));
dhm2dxout  = kron(dhm2dxoutv,ones(ny,1));


% Find the correction to the deviatoric stresses
Sxx2min  = -0.5*y.^2.*dsigma0in.^2;
Sxx2mout = -0.5*y.^2.*dsigma0out.^2;
SxxM2    = (x<=xN).*Sxx2min + (x>xN).*Sxx2mout;
Sxx2cin  = -0.5/beta*(y.^2.*dsigma0in.^2  - 4*(beta-1)*alpha*y.*dhdx.*dsigma0in  + 4*(beta-1)^2*alpha^2*dhdx.^2);
Sxx2cout = -0.5/beta*(y.^2.*dsigma0out.^2 - 4*(beta-1)*alpha*y.*dhdx.*dsigma0out + 4*(beta-1)^2*alpha^2*dhdx.^2);
SxxC2    = (x<=xN).*Sxx2cin + (x>xN).*Sxx2cout;
Sxx2     = (y<=hm).*SxxM2 + (y>hm).*SxxC2;
Syy2     = -Sxx2;

% Solve for the yy-stress correction 
RHSin  = -hv.^3.*dddsigma0inv/6  + 0.5*muOD*ddsigma0inv .*hv.^2 + muOD*dhdxv.^2.*(sigmayy0inv +2*beta)+(dhdxv+2*muOD).*hv.*dhdxv.*dsigma0inv  + hv.^2.*dhdxv.*dsigma0inv .^2/beta+2*hv.^3.*dsigma0inv .*ddsigma0inv/3/beta ...
    + 2*(beta-1)*dhm2dxinv  + alpha^3/beta*(beta-1)*(4*hv.*dhdxv.^2.*dsigma0inv  + hv.^2.*dhdxv.*dsigma0inv .^2 - 2*(beta-2)*dhdxv.^3)  ...
    - 2*alpha*(beta-1)*((1-2*(beta-1)*alpha/beta)*dhdxv.^3 - (alpha-2)*muOD*dhdxv.^2 - (alpha-1)*muOD*hv.*d2hdx2v) ...
    + alpha*(1-alpha)*(beta-1)/beta*((1-alpha)*beta*hv.^2.*d3hdx3v - 2*alpha*(4-beta)*hv.*dhdxv.*d2hdx2v) + 2*(beta-1)/beta*alpha^3*hv.^3.*dsigma0inv.*ddsigma0inv/3 ...
    - 2*(beta-1)/beta*alpha*(1-alpha^2)*hv.^2.*(d2hdx2v.*dsigma0inv  +dhdxv.*ddsigma0inv ) - 4*alpha*(beta-1)/beta*hv.*dhdxv.^2.*dsigma0inv;
RHSout = -hv.^3.*dddsigma0outv/6 - 0.5*muOD*ddsigma0outv.*hv.^2 - muOD*dhdxv.^2.*(sigmayy0outv+2*beta)+(dhdxv-2*muOD).*hv.*dhdxv.*dsigma0outv + hv.^2.*dhdxv.*dsigma0outv.^2/beta+2*hv.^3.*dsigma0inv .*ddsigma0inv/3/beta ...
    + 2*(beta-1)*dhm2dxoutv + alpha^3/beta*(beta-1)*(4*hv.*dhdxv.^2.*dsigma0outv + hv.^2.*dhdxv.*dsigma0outv.^2 - 2*(beta-2)*dhdxv.^3)  ...
    - 2*alpha*(beta-1)*((1-2*(beta-1)*alpha/beta)*dhdxv.^3 + (alpha-2)*muOD*dhdxv.^2 + (alpha-1)*muOD*hv.*d2hdx2v) ...
    + alpha*(1-alpha)*(beta-1)/beta*((1-alpha)*beta*hv.^2.*d3hdx3v - 2*alpha*(4-beta)*hv.*dhdxv.*d2hdx2v) + 2*(beta-1)/beta*alpha^3*hv.^3.*dsigma0outv.*ddsigma0outv/3 ...
    - 2*(beta-1)/beta*alpha*(1-alpha^2)*hv.^2.*(d2hdx2v.*dsigma0outv +dhdxv.*ddsigma0outv) - 4*alpha*(beta-1)/beta*hv.*dhdxv.^2.*dsigma0outv; 
% Boundary condition comes from integrating sigma2m in y 
sigma2in0  = ((beta-1)/beta*alpha^3/3 + 1.0/3/beta)*dsigma0inv(1)^2*hv(1)^2  - ddsigma0inv(1)*hv(1)^2/6  + (beta-1)*alpha*(1-alpha)^2*hv(1)*d2hdx2v(1)  ...
    - 2*(beta-1)/beta*alpha^2*(1-alpha)*(beta-2)*dhdxv(1)^2 - 2*(beta-1)/beta*alpha*(1-alpha^2)*hv(1)*dhdxv(1)*dsigma0inv(1);
sigma2out0 = ((beta-1)/beta*alpha^3/3 + 1.0/3/beta)*dsigma0outv(1)^2*hv(1)^2 - ddsigma0outv(1)*hv(1)^2/6 + (beta-1)*alpha*(1-alpha)^2*hv(1)*d2hdx2v(1)  ...
    - 2*(beta-1)/beta*alpha^2*(1-alpha)*(beta-2)*dhdxv(1)^2 - 2*(beta-1)/beta*alpha*(1-alpha^2)*hv(1)*dhdxv(1)*dsigma0outv(1);
% Now solve the ODE for yy stress
% Solve the ODE in the entrance
[xsolin, sig2solin] = ode45(@(x,sigma2) odefun2in(x,sigma2,RHSin,xv,muOD,red,rR,ell,h0),[0,1],sigma2in0);
% Solve the ODE in the exit
[xsolout, sig2solout] = ode45(@(x,sigma2) odefun2out(x,sigma2,RHSout,xv,muOD,red,rR,ell,h0),[1,0],sigma2out0);
% Interpolate back onto the solution grid
sigma2minv  = interp1(xsolin,sig2solin,xv);
sigma2moutv = interp1(xsolout,sig2solout,xv);
sigma2min   = kron(sigma2minv,ones(ny,1));
sigma2mout  = kron(sigma2moutv,ones(ny,1));

% Compute sigmayy2
sigmayym2in  = sigma2min  + 0.5*y.^2.*ddsigma0in;
sigmayym2out = sigma2mout + 0.5*y.^2.*ddsigma0out;
sigmayyM2    = (x<=xN).*sigmayym2in + (x>xN).*sigmayym2out;
sigmayyc2in  = sigma2min + 0.5*y.^2.*ddsigma0in  - 2*alpha*(beta-1)*(d2hdx2.*y - alpha*(h.*d2hdx2+dhdx.^2));
sigmayyc2out = sigma2mout  + 0.5*y.^2.*ddsigma0out - 2*alpha*(beta-1)*(d2hdx2.*y - alpha*(h.*d2hdx2+dhdx.^2));
sigmayyC2    = (x<=xN).*sigmayyc2in + (x>xN).*sigmayyc2out;
sigmayy2     = (y<=hm).*sigmayyM2 + (y>hm).*sigmayyC2;

% Find the xx-stress 
sigmaxxM2    = sigmayyM2 + 2*SxxM2;
sigmaxxC2    = sigmayyC2 + 2*SxxC2;
sigmaxx2     = (y<=hm).*sigmaxxM2 + (y>hm).*sigmaxxC2;

% Find the derivative of sigma2
dsigma2minv  = (RHSin  + muOD*sigma2minv)./hv;
dsigma2min   = kron(dsigma2minv,ones(ny,1));
dsigma2moutv = (RHSout - muOD*sigma2moutv)./hv;
dsigma2mout  = kron(dsigma2moutv,ones(ny,1));

% Compute the shear stress correction 
SxyM2in  = ((4*dsigma0in .*ddsigma0in  - dddsigma0in )/6.*y.^3-y.*dsigma2min )/muOD;
SxyM2out = ((4*dsigma0out.*ddsigma0out - dddsigma0out)/6.*y.^3-y.*dsigma2mout)/muOD;
SxyM2    = (x<=xN).*SxyM2in + (x>xN).*SxyM2out;
SxyC2in  = ((4*dsigma0in .*ddsigma0in  - dddsigma0in )/6*alpha^3.*h.^3 + 2*(beta-1)*dhm2dxin  + alpha^3*(beta-1)/beta*(4*h.*dhdx.^2.*dsigma0in  + h.^2.*dhdx.*dsigma0in .^2-2*(beta-2)*dhdx.^3) ...
    - y.*dsigma2min  - dddsigma0in /6.*(y.^3-alpha^3.*h.^3) + (beta-1)*alpha*d3hdx3.*(y.^2-alpha^2.*h.^2) - 2*(beta-1)*alpha^2*(d3hdx3.*h-dhdx.*d2hdx2).*(y-alpha*h) ...
    - 12*alpha^2*(beta-1)*dhdx.*d2hdx2.*(y-alpha*h) ...
    + 1.0/beta*(2.0/3*dsigma0in .*ddsigma0in .*(y.^3-alpha^3.*h.^3)-2*(beta-1)*alpha*(d2hdx2.*dsigma0in +dhdx.*ddsigma0in ).*(y.^2-alpha^2*h.^2) + 8*(beta-1)^2*alpha^2*dhdx.*d2hdx2.*(y-alpha*h)))/muOD;
SxyC2out = ((4*dsigma0out.*ddsigma0out - dddsigma0out)/6*alpha^3.*h.^3 + 2*(beta-1)*dhm2dxout + alpha^3*(beta-1)/beta*(4*h.*dhdx.^2.*dsigma0out + h.^2.*dhdx.*dsigma0out.^2-2*(beta-2)*dhdx.^3) ...
    - y.*dsigma2mout - dddsigma0out/6.*(y.^3-alpha^3.*h.^3) + (beta-1)*alpha*d3hdx3.*(y.^2-alpha^2.*h.^2) - 2*(beta-1)*alpha^2*(d3hdx3.*h-dhdx.*d2hdx2).*(y-alpha*h) ...
    - 12*alpha^2*(beta-1)*dhdx.*d2hdx2.*(y-alpha*h) ...
    + 1.0/beta*(2.0/3*dsigma0out.*ddsigma0out.*(y.^3-alpha^3.*h.^3)-2*(beta-1)*alpha*(d2hdx2.*dsigma0out+dhdx.*ddsigma0out).*(y.^2-alpha^2*h.^2) + 8*(beta-1)^2*alpha^2*dhdx.*d2hdx2.*(y-alpha*h)))/muOD;
SxyC2    = (x<=xN).*SxyC2in + (x>xN).*SxyC2out;
Sxy2     = (y<=hm).*SxyM2 + (y>hm).*SxyC2;

% Compute force and torque
% First find the stress components on the roll surface
hv = min(hv,1);
sigmaxx0Ch = interp2(x,y,sigmaxx0C,xv,hv);
sigmaxx2Ch = interp2(x,y,sigmaxxC2,xv,hv);
sigmayy0Ch = interp2(x,y,sigmayy0,xv,hv);
sigmayy2Ch = interp2(x,y,sigmayyC2,xv,hv);
Sxy0Ch     = interp2(x,y,Sxy0C,xv,hv);
Sxy2Ch     = interp2(x,y,SxyC2,xv,hv);

% Compute the integrands 
forceint  = sigmayy0Ch + delta^2*(sigmayy2Ch-0.5*dhdxv.^2.*sigmayy0Ch - muOD*dhdxv.*Sxy0Ch);
torqueint = muOD*Sxy0Ch - dhdxv.*(sigmaxx0Ch-sigmayy0Ch) + delta^2*(muOD*Sxy2Ch - ...
    2*muOD*dhdxv.^2.*Sxy0Ch - dhdxv.*(sigmaxx2Ch-sigmayy2Ch)+dhdxv.^3.*(sigmaxx0Ch-sigmayy0Ch));
SMforceint = sigmayy0Ch;
SMtorqueint = muOD*Sxy0Ch - dhdxv.*(sigmaxx0Ch-sigmayy0Ch);

% Use TRAPZ to find the integrals
% Force in N, Torque in Nm (hence the factor of 1000)
force  = -trapz(xv,forceint)*ww*ell*kM;
torque = trapz(xv,torqueint)*ww*ell*kM*delta*rR/1000; 
SMforce  = -trapz(xv,SMforceint)*ww*ell*kM;
SMtorque = trapz(xv,SMtorqueint)*ww*ell*kM*delta*rR/1000;

% Create a struct for output
% Constains solution grid, xN, then leading-order and corrected values for
% velocity (u,v), stress (sigmaxx, sigmayy, Sxy), force and torque.
result = struct;
result.x           = ell*x;
result.y           = h0*y;
result.u0          = angvel*rR*u0;
result.ucorr       = angvel*rR*(u0 + delta^2*u2);
result.v0          = angvel*rR*delta*v0;
result.vcorr       = angvel*rR*delta*(v0 + delta^2*v2);
result.sigmaxx0    = kM*sigmaxx0;
result.sigmaxxcorr = kM*(sigmaxx0 + delta^2*sigmaxx2); 
result.sigmayy0    = kM*(sigmayy0);
result.sigmayycorr = kM*(sigmayy0 + delta^2*sigmayy2);
result.Sxy0        = mu*kM*Sxy0;
result.Sxycorr     = mu*kM*(Sxy0 + delta^2*Sxy2);
result.force0      = SMforce;
result.forcecorr   = force;
result.torque0     = SMtorque;
result.torquecorr  = torque;
result.xN          = ell*xN;
end

%% h 
function res = hfunc(x,r,R,ell,h0)
% Evaluates roll gap thickness h at location x
res = 1-r*(2*x-x.^2);
end

%% h'
function res = dhdxfunc(x,r,R,ell,h0)
% Evaluates dhdx at location x
res = 2*r*(x-1);
end

%% h''
function res = d2hdx2func(x,r,R,ell,h0)
% Evaluates d2hdx2 at location x
res = 2*r*ones(size(x));
end

%% h'''
function res = d3hdx3func(x,r,R,ell,h0)
% Evaluates d2hdx2 at location x
res = zeros(size(x));
end

%% ODE function for sigmayy0in
function res = odefun0in(x,sigma0,alpha,beta,muOD,red,rR,ell,h0)
%%% ODE function for leading order entrance stress
% Interpolate h and h' at x
h      = hfunc(x,red,rR,ell,h0);
dhdx   = dhdxfunc(x,red,rR,ell,h0);
d2hdx2 = d2hdx2func(x,red,rR,ell,h0);
% Compute dsigma0/dx
res = 2*(alpha*beta-beta-alpha)*dhdx/h + muOD*sigma0/h;
end

%% ODE function for sigmayy0out
function res = odefun0out(x,sigma0,alpha,beta,muOD,red,rR,ell,h0)
%%% ODE function for leading order exit stress
% Interpolate h and h' at x
h      = hfunc(x,red,rR,ell,h0);
dhdx   = dhdxfunc(x,red,rR,ell,h0);
d2hdx2 = d2hdx2func(x,red,rR,ell,h0);
% Compute dsigma0/dx
res = 2*(alpha*beta-beta-alpha)*dhdx/h - muOD*sigma0/h;
end

%% ODE function for sigmayy2min
function res = odefun2in(x,sigma2m,RHSin,xv,muOD,red,rR,ell,h0)
%%% ODE function for entrance stress correction
% Interpolate h and RHS at x
h      = hfunc(x,red,rR,ell,h0);
RHS    = interp1(xv,RHSin,x);
% Compute dsigma2m/dx
res = RHS./h + muOD.*sigma2m./h;
end

%% ODE function for sigmayy2mout
function res = odefun2out(x,sigma2m,RHSout,xv,muOD,red,rR,ell,h0)
%%% ODE function for exit stress correction
% Interpolate h and h' at x
h      = hfunc(x,red,rR,ell,h0);
RHS    = interp1(xv,RHSout,x);
% Compute dsigma2m/dx
res = RHS./h - muOD.*sigma2m./h;
end

%% Function for finding xN
function res = xNfunc(x,xsolin,sig0solin,xsolout,sig0solout)
%%% Computes the difference between sig0in and sig0out 
%%% as a function of x
% Interpolate sig0in and sig0out
sigin  = interp1(xsolin,sig0solin,x);
sigout = interp1(xsolout,sig0solout,x);
% Compute the difference
res = sigin-sigout;
end

