#!/usr/bin/octave -q

% Copyright (c) 2016 Ed Brambley <E.J.Brmabley@damtp.cam.ac.uk>.
% All rights reserved.

0;

source 'derivatives.m';
source 'filter.m';
source 'integrator.m';

% Damping profile
function kp = damping(N, L, str, x0, x1, d)

  dx = L./(N+1);
  kp = zeros(1,N+1);
  
  % If there is no smoothing, use an exact method (Note: x0 and x1 cannot wrap)
  if (d == 0)
    n0 = round(x0./dx)+1; % Index of first point to damp
    n1 = round(x1./dx)+1; % Index of first point not to damp (gives damping for damping_x0 <= x < damping_x1)
    kp(n0:(n1-1)) = str;
    damping_length = (n1-n0).*dx; % Calculate the actual numerical damping length
    return;
  end

  % Use a bump function with half-width d
  i0 = floor((x0-d)/dx); % Start location
  i1 = ceil((x1+d)/dx); % End location
  for i = i0:i1 % Note: i is zero indexed
    x = i.*dx; % Location
    in = mod(i,N+1)+1; % Calculate wrapped index (one indexed)

    % Calculate amplitude
    if (x <= x0-d || x >= x1+d)
      A = 0;
    elseif (x < x0+d)
      y = 0.5*(1+(x-x0)./d); % y in (0,1)
      A = bump(y); % Smooth transition
    elseif (x > x1-d)
      y = 0.5*(1-(x-x1)./d); % y in (0,1)
      A = bump(y); % Smooth transition
    else
      A = 1;
    end

    kp(in) = str.*A;
  end

end


% Physical simulation parameters
L = 24;
T = 24;
x0 = 4;
x1 = 16;
smooth_d = 4;
direction = 1;
omega = 2.*pi;
damping_k = 3;
damping_x0 = 21;
damping_x1 = 23;
damping_d = 0.5;

% Numerical simulation parameters
N=3455;
Type="MO7";
Lm = Lp = 0;
Nm = Np = 3;
integrator = @lddrk56;
CFL=0.8;
tol=1e-8;
filter = @filter_s7;
S = 1;

% Perform computation
fprintf(stderr(), "Computing solution... ");
[B,D] = derivative_matrices_2d_circular(N, Lm, Lp, Nm, Np, Type);
v0 = wave_initial(N, L, x0, x1, omega, smooth_d, direction);
kp = damping(N, L, damping_k, damping_x0, damping_x1, damping_d);
kq = kp;

if (S < 0)
  [v,t,nt] = solve_wave_custom_integrator(B, D, N, v0, L, kp, kq, CFL.*L./(N+1), T, tol, integrator, @(x,dt) filter_intermittent(x, dt, @(x,str) filter_2d(x, filter, str  ), 1, -S));
else
  [v,t,nt] = solve_wave_custom_integrator(B, D, N, v0, L, kp, kq, CFL.*L./(N+1), T, tol, integrator, @(x,dt)                                     filter_2d(x, filter, S.*dt)        );
end
fprintf(stderr(), "done\n");

% Write real data, with one column to a line (i.e. [1,2,3;4,5,6] becomes 1 4\n2 5\n3 6\n)
function output(filename, matrix)
  fid = fopen(filename, 'w');
  fprintf(fid, [ repmat("%24.16g ", [1, rows(matrix)]) , "\n"], matrix);
  fclose(fid);
end

% Output
fprintf(stderr(), "Outputting text files... ");
x=(0:N).*L./(N+1);
skip = 6; % Gives 30 fps with above parameters
frame=0;
offset=1; % Used to avoid flickering when we get absolute zeros on a log scale.
for i=1:skip:length(t)
  y = reshape(v(i,:),2,N+1)(1,:);
  output(sprintf("movie/frame_%03d.txt", frame), [x;y]);
  frame = frame + 1;
end
fprintf(stderr(), "done\n");
