#!/usr/bin/octave -q

% Copyright (c) 2016 Ed Brambley <E.J.Brmabley@damtp.cam.ac.uk>.
% All rights reserved.

0;

source 'derivatives.m';
source 'filter.m';
source 'integrator.m';
source 'bump.m';
source 'problem.m';


% Helper function for checking
function ret = check_and_warn(A, B, desc, idx)
  ret = 0;
  if (A ~= B)
    fprintf(stderr, "Warning: index %d: %s doesn't match (got %g, expecting %g).\n", idx, desc, A, B)
    ret = 1;
  end
end

% Go from 2 ppw to 2^qm ppw, with 2^qs points per octave (not necessarily equally logarithmically spaced).
function N = equal_octave_N(qm, qs, L)

  % Calculate number of points N.
  %N = round(L*2.^(1:(2.^(-qs)):qm))-1;
  N = [2, reshape((1 + (1:(2.^qs)).'.*2.^(-qs)) * 2.^(1:(qm-1)), 1, 2.^qs.*(qm-1))]; % N is now the number of points per wavelength.
  N = round(L*N - 1); % Convert points per wavelength to number of points.
  N = unique(N); % Remove duplicates
end


% Parameters
L = 24;
qm = 7;
qs = 5;
N = equal_octave_N(qm, qs, L);

% Fig 1
Type       = { "MO7", "DRP7", "TSDRP", "MO15", "DRP15", "KL2", "KL4", "KL6", "KL8", "BBo9p", "BBo11p", "BBo13p" };
Lp         = [   0  ,   0   ,    0   ,   0   ,    0   ,   1  ,   1  ,   1  ,   1  ,    0   ,    0    ,    0     ];
Np         = [   3  ,   3   ,    3   ,   7   ,    7   ,   3  ,   3  ,   3  ,   3  ,    4   ,    5    ,    6     ];
S          = [  100 ,  150  ,   100  ,   2   ,    2   ,  100 ,   1  ,   1  ,   1  ,    8   ,    2    ,   1.5    ];
integrator = repmat({@rk45ode    }, 1, 12);
CFL        = repmat([  0.9       ], 1, 12);
filter     = repmat({@filter_s164}, 1, 12);

% Fig 2
Type       = {    Type{:}   , repmat({   "MO7"  ,  "TSDRP"  ,    "MO15"   ,    "DRP15"  ,    "KL2"   ,    "KL4"   }, 1, 5){:} };
Lp         = [       Lp     , repmat([     0    ,     0     ,      0      ,       0     ,      1     ,      1     ], 1, 5)    ];
Np         = [       Np     , repmat([     3    ,     3     ,      7      ,       7     ,      3     ,      3     ], 1, 5)    ];
S          = [       S      , repmat([     2    ,     2     ,      2      ,       2     ,      1     ,      1     ], 1, 5)    ];
filter     = {   filter{:}  , repmat({@filter_s7, @filter_s7, @filter_s164, @filter_s164, @filter_s82, @filter_s82}, 1, 5){:} };
integrator = { integrator{:}, repmat({@rk45ode}, 1, 6){:} , repmat({@rk4ode}, 1, 6){:} , repmat({@rk4ode}, 1, 6){:} , repmat({@lddrk56}, 1, 6){:} , repmat({@lddrk56}, 1, 6){:} };
CFL        = [      CFL     , repmat([0.9]     , 1, 6)    , repmat([0.2]    , 1, 6)    , repmat([0.8]    , 1, 6)    , repmat([0.2]     , 1, 6)    , repmat([0.8]     , 1, 6)    ];

% Fig 3
Type       = {    Type{:}   ,    "BBo9p"   ,    "BBo9p"   ,    "BBo11p"   ,    "BBo11p"   ,    "BBo13p"   ,    "BBo13p"    };
Lp         = [       Lp     ,       0      ,       0      ,       0       ,       0       ,       0       ,       0        ];
Np         = [       Np     ,       4      ,       4      ,       5       ,       5       ,       6       ,       6        ];
CFL        = [      CFL     ,      0.2     ,      0.8     ,      0.2      ,      0.8      ,      0.2      ,      0.8       ];
filter     = {    filter{:} , @filter_bbo9p, @filter_bbo9p, @filter_bbo11p, @filter_bbo11p, @filter_bbo13p, @filter_bbo13p };
S          = [       S      , repmat([  2.4 ], 1, 6)    ];
integrator = { integrator{:}, repmat({@bbo6s}, 1, 6){:} };

% Fig 3 KL
Type       = {    Type{:}   ,     "KL2"    ,     "KL2"    ,     "KL4"     ,     "KL4"     ,     "KL6"     ,     "KL8"      };
Lp         = [       Lp     ,       1      ,       1      ,       1       ,       1       ,       1       ,       1        ];
Np         = [       Np     ,       3      ,       3      ,       3       ,       3       ,       3       ,       3        ];
integrator = { integrator{:},    @rk4ode   ,   @lddrk56   ,    @rk4ode    ,    @lddrk56   ,    @lddrk56   ,    @lddrk56    };
CFL        = [      CFL     , repmat([    0.8    ], 1, 6)    ];
S          = [       S      , repmat([     1     ], 1, 6)    ];
filter     = {   filter{:}  , repmat({@filter_s82}, 1, 6){:} };

% Number of parameter combinations, and the number of digits needed for this number of combinations
if (columns(Type) ~= columns(Lp        )) fprintf(stderr(), "Error: Type doesn't match Lp.\n"        ); exit; end
if (columns(Type) ~= columns(Np        )) fprintf(stderr(), "Error: Type doesn't match Np.\n"        ); exit; end
if (columns(Type) ~= columns(S         )) fprintf(stderr(), "Error: Type doesn't match S.\n"         ); exit; end
if (columns(Type) ~= columns(filter    )) fprintf(stderr(), "Error: Type doesn't match filter.\n"    ); exit; end
if (columns(Type) ~= columns(integrator)) fprintf(stderr(), "Error: Type doesn't match integrator.\n"); exit; end
if (columns(Type) ~= columns(CFL       )) fprintf(stderr(), "Error: Type doesn't match CFL.\n"       ); exit; end
total = columns(N).*columns(Type);
digs = length(sprintf("%d", total));
prog = regexprep(program_name(), "\\.m$", "");

% Process argument (if present)
if (nargin < 1)

   printf("%d\n", total);

elseif (strcmp(argv(){1}, "output"))
    
  % Write text files
  load (sprintf("%s.mat"  , prog));

  % Extract sup norm errors
  err = permute(err, [3, 2, 1])(:,:,2); % So now err(Ti, Ni) is the storage format

  % Fig 1
  fid = fopen('fig1.txt', 'w');
  fprintf(fid, "# Integrator = rk45\n");
  fprintf(fid, "# Filter = s164\n");
  fprintf(fid, "# Err = Esup\n");
  fprintf(fid, "# Filter Strengths = given in header\n");
  fprintf(fid, "#\n# <N+1> <MO7(%g)> <DRP7(%g)> <TSDRP(%g)> <MO15(%g)> <DRP15(%g)> <KL2(%g)> <KL4(%g)> <KL6(%g)> <KL8(%g)> <BB9(%g)> <BB11(%g)> <BB13(%g)>\n\n", S(1:12));
  fprintf(fid, [ repmat('%24.16g ', [1, 13]) , '\n'], [ N; err(1:12,:) ]);
  fclose(fid);

  % Fig 2
  fid = fopen('fig2a.txt', 'w');
  Ti=13;
  fprintf(fid, "# Scheme = %s\n", Type{Ti});
  fprintf(fid, "# Filter = %s (S=%g)\n", functions(filter{Ti}).function, S(Ti));
  fprintf(fid, "# Err = Esup\n");
  fprintf(fid, "#\n# <N+1> <RK45> <RK4(0.2)> <RK4(0.8)> <LDDRK56(0.2)> <LDDRK56(0.8)>\n\n");
  fprintf(fid, [ repmat('%24.16g ', [1, 6]) , '\n'], [ N; err((0:4).*6+Ti,:)]);
  fclose(fid);
  
  fid = fopen('fig2b.txt', 'w');
  Ti=14;
  fprintf(fid, "# Scheme = %s\n", Type{Ti});
  fprintf(fid, "# Filter = %s (S=%g)\n", functions(filter{Ti}).function, S(Ti));
  fprintf(fid, "# Err = Esup\n");
  fprintf(fid, "#\n# <N+1> <RK45> <RK4(0.2)> <RK4(0.8)> <LDDRK56(0.2)> <LDDRK56(0.8)>\n\n");
  fprintf(fid, [ repmat('%24.16g ', [1, 6]) , '\n'], [ N; err((0:4).*6+Ti,:)]);
  fclose(fid);
  
  fid = fopen('fig2c.txt', 'w');
  Ti=15;
  fprintf(fid, "# Scheme = %s\n", Type{Ti});
  fprintf(fid, "# Filter = %s (S=%g)\n", functions(filter{Ti}).function, S(Ti));
  fprintf(fid, "# Err = Esup\n");
  fprintf(fid, "#\n# <N+1> <RK45> <RK4(0.2)> <RK4(0.8)> <LDDRK56(0.2)> <LDDRK56(0.8)>\n\n");
  fprintf(fid, [ repmat('%24.16g ', [1, 6]) , '\n'], [ N; err((0:4).*6+Ti,:)]);
  fclose(fid);
  
  fid = fopen('fig2d.txt', 'w');
  Ti=16;
  fprintf(fid, "# Scheme = %s\n", Type{Ti});
  fprintf(fid, "# Filter = %s (S=%g)\n", functions(filter{Ti}).function, S(Ti));
  fprintf(fid, "# Err = Esup\n");
  fprintf(fid, "#\n# <N+1> <RK45> <RK4(0.2)> <RK4(0.8)> <LDDRK56(0.2)> <LDDRK56(0.8)>\n\n");
  fprintf(fid, [ repmat('%24.16g ', [1, 6]) , '\n'], [ N; err((0:4).*6+Ti,:)]);
  fclose(fid);
  
  fid = fopen('fig2e.txt', 'w');
  Ti=17;
  fprintf(fid, "# Scheme = %s\n", Type{Ti});
  fprintf(fid, "# Filter = %s (S=%g)\n", functions(filter{Ti}).function, S(Ti));
  fprintf(fid, "# Err = Esup\n");
  fprintf(fid, "#\n# <N+1> <RK45> <RK4(0.2)> <RK4(0.8)> <LDDRK56(0.2)> <LDDRK56(0.8)>\n\n");
  fprintf(fid, [ repmat('%24.16g ', [1, 6]) , '\n'], [ N; err((0:4).*6+Ti,:)]);
  fclose(fid);
  
  fid = fopen('fig2f.txt', 'w');
  Ti=18;
  fprintf(fid, "# Scheme = %s\n", Type{Ti});
  fprintf(fid, "# Filter = %s (S=%g)\n", functions(filter{Ti}).function, S(Ti));
  fprintf(fid, "# Err = Esup\n");
  fprintf(fid, "#\n# <N+1> <RK45> <RK4(0.2)> <RK4(0.8)> <LDDRK56(0.2)> <LDDRK56(0.8)>\n\n");
  fprintf(fid, [ repmat('%24.16g ', [1, 6]) , '\n'], [ N; err((0:4).*6+Ti,:)]);
  fclose(fid);
  
  % Fig 3
  fid = fopen('fig3.txt', 'w');
  Ti = 43;
  fprintf(fid, "# Scheme = Filter\n");
  fprintf(fid, "# Filter Strength = 2.4\n");
  fprintf(fid, "# Integrator = BBo6s (given CFL)\n");
  fprintf(fid, "# Err = Esup\n");
  fprintf(fid, "#\n# <N+1> <BBo9p(0.2)> <BBo9p(0.8)> <BBo11p(0.2)> <BBo11p(0.8)> <BBo13p(0.2)> <BBo13p(0.8)>\n");
  fprintf(fid, [ repmat('%24.16g ', [1, 7]) , '\n'], [N; err(Ti:(Ti+5),:) ]);
  fclose(fid);

  % Fig 3
  fid = fopen('fig3_kl.txt', 'w');
  Ti = 49;
  fprintf(fid, "# Filter = S82(1.0)\n");
  fprintf(fid, "# CFL = 0.8\n");
  fprintf(fid, "# Err = Esup\n");
  fprintf(fid, "#\n# <N+1> <KL2(RK4)> <KL2(LDDRK56)> <KL4(RK4)> <KL4(LDDRK56)> <KL6(LDDRK56)> <KL8(LDDRK56)>\n");
  fprintf(fid, [ repmat('%24.16g ', [1, 7]) , '\n'], [N; err(Ti:end,:) ]);
  fclose(fid);

  % Problem description at highest resolution
  [ v0, kp, kq ] = damped_wave_setup(N(end),L);
  fid = fopen('problem.txt', 'w');
  fprintf(fid, "# <x> <p> <q> <kp> <kq>\n");
  fprintf(fid, [ repmat('%24.16g ', [1, 5]) , '\n'], ...
          [ (0:(N(end))).*L./(N(end)+1); ...
	    reshape(v0,2,N(end)+1)(1,:); ...
	    reshape(v0,2,N(end)+1)(2,:); ...
	    kp;
	    kq ]);
  fclose(fid);

elseif (strcmp(argv(){1}, "combine"))
    
    % Compose individual outputs into one

    err = zeros(3, columns(N), columns(Type));
    v   = cell(    columns(N), columns(Type));
    idx = 0;
      for Ti = 1:columns(Type)
	for Ni = 1:columns(N)
	  idx = idx + 1;
	  
	  % Load this dataset
	  load (sprintf("data/%s_%0*d.mat", prog, digs, idx));

	  % Check that this is what we were expecting
	  warn = 0;
	  warn = warn + check_and_warn(res.N   , N(Ni)           , "N"   , idx);
	  warn = warn + check_and_warn(res.S   , S(Ti)           , "S"   , idx);
	  warn = warn + check_and_warn(res.CFL , CFL(Ti)         , "CFL" , idx);
	  warn = warn + check_and_warn(res.Lm  , Lp(Ti)          , "Lm"  , idx);
	  warn = warn + check_and_warn(res.Lp  , Lp(Ti)          , "Lp"  , idx);
	  warn = warn + check_and_warn(res.Nm  , Np(Ti)          , "Nm"  , idx);
	  warn = warn + check_and_warn(res.Np  , Np(Ti)          , "Np"  , idx);
	  warn = warn + check_and_warn(res.Type, Type{Ti}        , "Type", idx);
	  if (~ strcmp(functions(res.integrator).function, functions(integrator{Ti}).function))
	    fprintf(stderr, "Warning: index %d: integrator doesn't match.\n", idx);
	    warn = warn + 1;
	  end
	  if (~ strcmp(functions(res.filter    ).function, functions(filter{Ti}    ).function))
	    fprintf(stderr, "Warning: index %d: filter doesn't match.\n"    , idx);
	    warn = warn + 1;
	  end
		  
	  if (warn > 0)
	    fprintf(stderr, "\n");
	    return;
	  end

	  % Add this dataset to the amalgamated data
	  err(:, Ni, Ti) = res.err;
	  v{     Ni, Ti} = res.v  ;
	end
      end

    % Save amalgamated data
    save ("-binary", sprintf("%s.mat"  , prog),      "err", "N", "Type", "Lp", "Np", "S", "integrator", "CFL", "filter");
    save ("-binary", sprintf("%s_v.mat", prog), "v", "err", "N", "Type", "Lp", "Np", "S", "integrator", "CFL", "filter");

else

  % Get this index
  idx = str2double(argv(){1});
  if (~(idx >= 1 && idx <= total))
    fprintf(stderr, "Error: index \"%s\" should be between 1 and %d\n", argv(){1}, total);
    return;
  end
  
  % Get parameters for this index
  Ni = 1 +     mod(idx-1, columns(N));
  Ti = 1 + idivide(idx-1, columns(N), "fix");
	       
  res.N          =          N(Ni);
  res.S          =          S(Ti);
  res.CFL        =        CFL(Ti);
  res.Lp         =         Lp(Ti);
  res.Lm         =     res.Lp;
  res.Np         =         Np(Ti);
  res.Nm         =     res.Np;
  res.Type       =       Type{Ti};
  res.filter     =     filter{Ti};
  res.integrator = integrator{Ti};

  % Perform computation
  [ res.err, res.v ] = damped_wave_test(res.N, res.Lm, res.Lp, res.Nm, res.Np, res.Type, res.integrator, res.CFL, res.filter, res.S);

  % Save resuls
  save ("-binary", sprintf("data/%s_%0*d.mat", prog, digs, idx), "res");

end
