% Copyright (c) 2016 Ed Brambley <E.J.Brmabley@damtp.cam.ac.uk>.
% All rights reserved.

0;

% Damping profile
function [ kp, damping_length ] = damping(N, L, str, x0, x1, d)

  dx = L./(N+1);
  kp = zeros(1,N+1);
  
  % If there is no smoothing, use an exact method (Note: x0 and x1 cannot wrap)
  if (d == 0)
    n0 = round(x0./dx)+1; % Index of first point to damp
    n1 = round(x1./dx)+1; % Index of first point not to damp (gives damping for damping_x0 <= x < damping_x1)
    kp(n0:(n1-1)) = str;
    damping_length = (n1-n0).*dx; % Calculate the actual numerical damping length
    return;
  end

  % Use a bump function with half-width d
  i0 = floor((x0-d)/dx); % Start location
  i1 = ceil((x1+d)/dx); % End location
  for i = i0:i1 % Note: i is zero indexed
    x = i.*dx; % Location
    in = mod(i,N+1)+1; % Calculate wrapped index (one indexed)

    % Calculate amplitude
    if (x <= x0-d || x >= x1+d)
      A = 0;
    elseif (x < x0+d)
      y = 0.5*(1+(x-x0)./d); % y in (0,1)
      A = bump(y); % Smooth transition
    elseif (x > x1-d)
      y = 0.5*(1-(x-x1)./d); % y in (0,1)
      A = bump(y); % Smooth transition
    else
      A = 1;
    end

    kp(in) = str.*A;
  end

  % Since bump function have average value 1/2, and are symmetric about x0 and x1, the total damping length is x1-x0;
  damping_length = x1-x0;
end


% Harmonic initial conditions for left or right going waves.
function v = wave_initial(N, L, ... % Number of points and length of grid
			  x0,x1, ... % Start and end position of wave
			  omega, ... % Wavenumber of wave
			  d, ... % Extra length added to start and end of wave for smooth start
			  direct) % Direction: +1 = right propagating, -1 = left propagating
  v = zeros(2*N+2,1);

  dx = L./(N+1);
  i0 = floor((x0-d)/dx); % Start location
  i1 = ceil((x1+d)/dx); % End location
  for i = i0:i1
    x = i.*dx; % Location
    in = mod(i,N+1)+1; % Calculate wrapped index

    % Calculate wave amplitude
    if (x <= x0-d || x >= x1+d)
      A = 0;
    elseif (x < x0)
      y = 1 + (x-x0)./d; % y in (0,1)
      A = bump(y); % Smooth transition
    elseif (x > x1)
      y = 1 - (x-x1)./d; % y in (0,1)
      A = bump(y); % Smooth transition
    else
      A = 1;
    end
    
    % Modulate to become a wave
    A .*= cos(omega.*(x-x0));

    % q = p for right propagating, q = -p for left propagating
    v(2*in-1) +=        A;
    v(2*in  ) += direct*A;
  end
end


function ret = capat(val, cap)
  if (val > cap)
    ret = cap;
  else
    ret = val;
  end
end

% Helper function for time integration below
function vdot = solve_wave_xdot(v, t, B, D, N, kp, kq)
  % Calculate:
  %  p_t = - q_x - kp p
  %  q_t = - p_x - kq q
  % where v = (p_0, q_0, ..., p_N, q_N)

  % Sanity check
  if (size(v)(2) ~= 1)
    fprintf(stderr(), "solve_wave_xdot: Warning: v is not a column vector!\n");
    v = v(:);
  end
  
  % Calculate v_x from B v_x = D v
  vdot = B\(D*v);
	    
  % Separate out p and q into two rows
  vdot = reshape(vdot, 2, N+1);
  v    = reshape(v   , 2, N+1);

  % Swap rows in v_x (and switch sign), and then calculate vdot
  vdot([1, 2], :) = -vdot([2, 1], :);
  vdot -= [kp; kq] .* v;
  vdot = vdot(:);
end


% Solve the wave equation with given initial conditions and using a custom integrator and filter
function [v,t,nt] = solve_wave_custom_integrator(B, ... % \beta_j 2D matrix 2N+2 x 2N+2
						 D, ... % d_n 2D matrix, 2N+2 x 2N+2
						 N, ... % points are 0, ..., N
						 v0, ... % Initial conditions, column vector size 2N+2
						 L, ... % Length of grid
						 kp, kq, ... % Pressure and momentum damping respectively, each row vectors of size N+1
						 dt, ... % Time increment
						 T, ... % Final time
						 tol, ... % Error tolerance
						 integrator, ... % Integrator
						 filter, ... % Filtering filter(x, dt)
						 varargin); % Optional flags

  % Solve the damped wave equation on a periodic grid (so don't apply boundary conditions):
  %  p_t + q_x = - kp p
  %  q_t + p_x = - kq q
  %
  % Solution vector v is given by (p_0, q_0, ..., p_N, q_N).


  % Since x_0 = x_{N+1} = both 0 and L, therefore delta X = L/(N+1)
  % Derivatives given by Bv' = Dv/(delta X)
  D = D.*(N+1)./L;

  % Check row and column nature of v0, kp and kq
  v0 = v0(:);
  kp = kp(:).';
  kq = kq(:).';

  % Solve
  [v,t,nt] = integrator(@(x,t) solve_wave_xdot(x,t,B,D,N,kp,kq), filter, v0, 0, dt, T, tol, varargin{:});
end


% Test function to be run
function [ err, vout ] = damped_wave_test(N,Lm,Lp,Nm,Np,Type,... % Spatial derivative parameters
					  integrator,... % Time integrator
					  CFL,... % CFL for time step
					  filter,... % Spatial filter
					  S); % Filtering strength.  If negative, filter every -S seconds.


  % Setup
  L = 24;
  T = L;
  tol = 1e-8;
  x0 = 4;
  x1 = 16;
  damping_k = 3;
  damping_x0 = 21;
  damping_x1 = 23;
  damping_d = 0.5; % How much to smooth damping.  Set to zero to get abrupt damping.
  smooth_d = 4;
  direction = 1;
  omega = 2.*pi;

  v0 = wave_initial(N, L, x0, x1, omega, smooth_d, direction);
  [ kp, damping_exponent ] = damping(N, L, damping_k, damping_x0, damping_x1, damping_d);
  damping_exponent = damping_exponent .* damping_k;
  kq = kp;


  [B,D] = derivative_matrices_2d_circular(N, Lm, Lp, Nm, Np, Type);

  if (S < 0)
    [v,t,nt] = solve_wave_custom_integrator(B, D, N, v0, L, kp, kq, CFL.*L./(N+1), T, tol, integrator, @(x,dt) filter_intermittent(x, dt, @(x,str) filter_2d(x, filter, str  ), 1, -S)  , "final");
  else
    [v,t,nt] = solve_wave_custom_integrator(B, D, N, v0, L, kp, kq, CFL.*L./(N+1), T, tol, integrator, @(x,dt)                                     filter_2d(x, filter, capat(S.*dt, 1)), "final");
  end
  v = v(end,:).'.* exp(damping_exponent);
  
  % Actual L2 and sup norm errors
  err(1) = sqrt(sumsq(v-v0)./(2*N+2));
  err(2) = max(abs(v-v0));
  err(3) = T./nt; % Average time step length (nt is the number of time steps taken)

  % Output v if requested
  if (nargout() > 1)
    vout = v;
  end
end
