% Copyright (c) 2016 Ed Brambley <E.J.Brmabley@damtp.cam.ac.uk>.
% All rights reserved.

0;

% Integrate an ODE forward in time using classic RK4, with filtering each step
function [v, t_array, nt ] = rk4ode(xdot, ... % Function xdot(x, t) giving derivative (column vector)
				    filter, ... % Function filter(x, dt) filtering x
				    x, ... % Initial value of x (column vector)
				    t, ... % Initial time
				    dt, ... % Time increment
				    T, ... % Final time
				    err, ... % Error tolerance (not used)
				    varargin); % Any additional flags

  % Sanity check : x should be a column vector
  if (size(x)(2) ~= 1)
    fprintf(stderr(), "rk4ode: Warning: x is not a column vector!\n");
    x = x(:);
  end

  % Flags:
  record = true; % Should we record each step?
  
  % Process additional flags
  for i = 1:length(varargin)
    switch (varargin{i})
      case "final"
	% Only return the final result, not the steps on the way
	record = false;
      otherwise
	fprintf(stderr(), "rk4ode: Warning: unknown flag \"%s\"\n", varargin{i});
    end
  end  

  % Preallocate space for v and t_array
  if (record)
    v = zeros(ceil(T./dt+1), rows(x));
    t_array = zeros(ceil(T./dt+1), 1);
    v(1,:) = x.';
    t_array(1) = t;
  end
  i = 1;

  while (t < T)

    % Are we taking the last step?
    if (t + dt >= T)
       dt = T - t;
       T = 0.5.*t-1; % Ensures T < t, and so ensures this is the last iteration
    end

    % RK4
    lst = xdot(x               , t          ); % k1
    sum = lst;
    lst = xdot(x + 0.5.*dt.*lst, t + 0.5.*dt); % k2
    sum = sum + 2.*lst;
    lst = xdot(x + 0.5.*dt.*lst, t + 0.5.*dt); % k3
    sum = sum + 2.*lst;
    lst = xdot(x +      dt.*lst, t +      dt); % k4
    sum = sum + lst;
  
%   x = x + dt./6.*(k1 + 2.*k2 + 2.*k3 + k4);
    x = x + dt./6.*sum;
    x = filter(x, dt);
    t = t + dt;
    i = i + 1;
    if (record)
      v(i,:) = x.';
      t_array(i) = t;
    end
  end

  if (~record)
     % Record the final step if we haven't already
     t_array(1) = t;
     v(1,:) = x.';
  end
  nt = i-1;

end


% Integrate an ODE forward in time using Cash-Carp RK45, as described in Numerical Recipes, with filtering each step
function [ v, t_array, nt ] = rk45ode(xdot, ... % Function xdot(x, t) giving derivative (column vector)
				      filter, ... % Function filter(x, dt) filtering x
				      x, ... % Initial value of x (column vector)
				      t, ... % Initial time
				      max_dt, ... % Initial and maximum time increment
				      T, ...  % Final time
				      err, ... % Error tolerance
				      varargin); % Any additional flags

  % Cash-Carp coefficient (Wikipedia and Numerical Recipes agree on these
  a2 = 1./5;
  a3 = 3./10;
  a4 = 3./5;
  a5 =  1;
  a6 = 7./8;

  c1 =  37./378;
  c2 =    0;
  c3 = 250./621;
  c4 = 125./594;
  c5 =    0;
  c6 = 512./1771;

  c1s =  2825./27648;
  c2s =      0;
  c3s = 18575./48384;
  c4s = 13525./55296;
  c5s =   277./14336;
  c6s =     1./4;

  b21 =     1./5;
  b31 =     3./40;
  b32 =     9./40;
  b41 =     3./10;
  b42 =    -9./10;
  b43 =     6./5;
  b51 =   -11./54;
  b52 =     5./2;
  b53 =   -70./27;
  b54 =    35./27;
  b61 =  1631./55296;
  b62 =   175./512;
  b63 =   575./13824;
  b64 = 44275./110592;
  b65 =   253./4096;

  % Safety factor
  S = 0.9;

  % Sanity check : x should be a column vector
  if (size(x)(2) ~= 1)
    fprintf(stderr(), "rk45ode: Warning: x is not a column vector!\n");
    x = x(:);
  end

  % Flags:
  record = true; % Should we record each step?
  
  % Process additional flags
  for i = 1:length(varargin)
    switch (varargin{i})
      case "final"
	% Only return the final result, not the steps on the way
	record = false;
      otherwise
	fprintf(stderr(), "rk45ode: Warning: unknown flag \"%s\"\n", varargin{i});
    end
  end  

  % Start where we're given
  if (record)
    v(1,:) = x.';
    t_array(1) = t;
  end
  i = 1;

  dt = max_dt;
  finished = false;
  do

    do

      % Check for too small a step size
      if (t + dt == t)
	fprintf(stderr(), "rk45ode: Error: step size below machine precision!\n");
	return;
      end

      % Don't go past T
      finished = false;
      if (t + dt > T)
	finished = true;
	dt = T - t;
      end

      % Attempt a step using RK45
      k1 = dt*xdot(x                                     , t         );
      k2 = dt*xdot(x + b21*k1                            , t + dt.*a2);
      k3 = dt*xdot(x + b31*k1+b32*k2                     , t + dt.*a3);
      k4 = dt*xdot(x + b41*k1+b42*k2+b43*k3              , t + dt.*a4);
      k5 = dt*xdot(x + b51*k1+b52*k2+b53*k3+b54*k4       , t + dt.*a5);
      k6 = dt*xdot(x + b61*k1+b62*k2+b63*k3+b64*k4+b65*k5, t + dt.*a6);
      inc = c1*k1 + c2*k2 + c3*k3 + c4*k4 + c5*k5 + c6*k6;
      delta = max(abs((c1-c1s)*k1 + (c2-c2s)*k2 + (c3-c3s)*k3 + (c4-c4s)*k4 + (c5-c5s)*k5 + (c6-c6s)*k6)); % Estimated error in this step
      delta0 = err.*min(max(abs(inc)), dt); % This is the target error (minimum of relative and absolute error when integrated all the way to time T)
      
      % Do we have a sufficiently accurate step?
      if (delta < delta0)
	break;
      end
      
      % Refine with a prediction of the correct smaller step size
      dt = S.*dt.*(delta0/delta).^0.25; % Suggested estimate of smaller step size by numerical recipes, with a safety factor just in case.

    until (false)

    % Make the step
    x = filter(x + inc, dt);
    t = t + dt;
    i = i + 1;
    if (record)
      v(i,:) = x.';
      t_array(i) = t;
    end

    % Update the step size, which is now too small
    % dt = min(S.*dt.*(delta0/delta).^0.2, max_dt); % This is the numerical recipes version, which allows for delta0 to be independent of dt.
    dt = min(S.*dt.*(delta0/delta).^0.25, max_dt); % This version assumes that delta0 is a linear function of dt (which it is).

  until (finished)

  if (~record)
     % Record the final step if we haven't already
     t_array(1) = t;
     v(1,:) = x.';
  end
  nt = i-1;

end


% Integrate an ODE foward in time using lsode, with filtering each given interval
function [ v, t_array, nt ] = lsode_filter(xdot, ... % Function xdot(x, t) giving derivative (column vector)
					   filter, ... % Function filter(x, dt) filtering x
					   x, ... % Initial value of x (column vector)
					   t, ... % Initial time
					   dt, ... % Time increment
					   T, ... % Final time
					   err, ... % Error tolerance
					   varargin); % Any additional flags

  % Sanity check : x should be a column vector
  if (size(x)(2) ~= 1)
    fprintf(stderr(), "lsode_filter: Warning: x is not a column vector!\n");
    x = x(:);
  end

  % Flags:
  record = true; % Should we record each step?
  
  % Process additional flags
  for i = 1:length(varargin)
    switch (varargin{i})
      case "final"
	% Only return the final result, not the steps on the way
	record = false;
      otherwise
	fprintf(stderr(), "lsode_filter: Warning: unknown flag \"%s\"\n", varargin{i});
    end
  end  

  if (record)
    % Preallocate space for v
    v = zeros(ceil(T./dt+1), rows(x));
    t_array = zeros(ceil(T./dt+1), 1);
    v(1,:) = x.';
    t_array(1) = t;
  end
  i = 1;

  while (t < T)

    % Are we taking the last step?
    if (t + dt >= T)
       dt = T - t;
       T = 0.5.*t-1; % Ensures T < t, and so ensures this is the last iteration
    end

    % lsode
    lsode_options("absolute tolerance", err);
    lsode_options("relative tolerance", err);
    [step, istate, msg] = lsode(xdot, x, [t, t+dt]);
    if (istate ~= 2)
      fprintf(stderr(), "solve_wave: lsode error %d: %s\n", istate, msg);
    end
    x = step(end,:).';
    x = filter(x, dt);
    t = t + dt;
    i = i + 1;
    if (record)
      v(i,:) = x.';
      t_array(i) = t;
    end
  end

  if (~record)
     % Record the final step if we haven't already
     t_array(1) = t;
     v(1,:) = x.';
  end
  nt = i-1;

end


% Integrate an ODE forward in time using Hu, Hussaini & Manthey's LDDRK56, with filtering each step
function [v, t_array, nt ] = lddrk56(xdot, ... % Function xdot(x, t) giving derivative (column vector)
				     filter, ... % Function filter(x, dt) filtering x
				     x, ... % Initial value of x (column vector)
				     t, ... % Initial time
				     dt, ... % Time increment
				     T, ... % Final time
				     err, ... % Error tolerance (not used)
				     varargin); % Any additional flags

  % Sanity check : x should be a column vector
  if (size(x)(2) ~= 1)
    fprintf(stderr(), "lddrk56: Warning: x is not a column vector!\n");
    x = x(:);
  end

  % Flags:
  record = true; % Should we record each step?
  
  % Process additional flags
  for i = 1:length(varargin)
    switch (varargin{i})
      case "final"
	% Only return the final result, not the steps on the way
	record = false;
      otherwise
	fprintf(stderr(), "lddrk56: Warning: unknown flag \"%s\"\n", varargin{i});
    end
  end  

  % LDDRK56 coefficients
  % Coefficients as given in table iii
  % a1 = b1 = 1;
  a2 = b2 = 1./2;
  a3 = b3 = 1./6;
  a4 = b4 = 1./24;
  a5 = 0.00361050;
  b5 = 0.0121101;
  b6 = 0.00285919;

  % Conversion of coefficients as given in (5.2)
  aa5 = a2;
  aa4 = a3./a2;
  aa3 = a4./a3;
  aa2 = a5./a4;
  % aa1 = 0;

  % bb1 = 0;
  bb6 = b2;
  bb5 = b3./b2;
  bb4 = b4./b3;
  bb3 = b5./b4;
  bb2 = b6./b5;

  if (record)
    % Preallocate space for v and t_array
    v = zeros(ceil(T./dt+1), rows(x));
    t_array = zeros(ceil(T./dt+1), 1);
    v(1,:) = x.';
    t_array(1) = t;
  end
  i = 1;

  while (t < T)

    % Are we taking the last two steps?
    if (t + 2.*dt >= T)
       dt = 0.5.*(T - t);
       T = 0.5.*t-1; % Ensures T < t, and so ensures this is the last iteration
    end

    % LDDRK5
    lst = dt.*xdot(x           , t          ); % k1
    lst = dt.*xdot(x + aa2.*lst, t + aa2.*dt); % k2
    lst = dt.*xdot(x + aa3.*lst, t + aa3.*dt); % k3
    lst = dt.*xdot(x + aa4.*lst, t + aa4.*dt); % k4
    lst = dt.*xdot(x + aa5.*lst, t + aa5.*dt); % k5
    x = x + lst;
    x = filter(x, dt);
    t = t + dt;
    i = i + 1;
    if (record)
      v(i,:) = x.';
      t_array(i) = t;
    end

    % LDDRK6
    lst = dt.*xdot(x           , t          ); % k1
    lst = dt.*xdot(x + bb2.*lst, t + bb2.*dt); % k2
    lst = dt.*xdot(x + bb3.*lst, t + bb3.*dt); % k3
    lst = dt.*xdot(x + bb4.*lst, t + bb4.*dt); % k4
    lst = dt.*xdot(x + bb5.*lst, t + bb5.*dt); % k5
    lst = dt.*xdot(x + bb6.*lst, t + bb6.*dt); % k6
    x = x + lst;
    x = filter(x, dt);
    t = t + dt;
    i = i + 1;
    if (record)
      v(i,:) = x.';
      t_array(i) = t;
    end

  end

  if (~record)
     % Record the final step if we haven't already
     t_array(1) = t;
     v(1,:) = x.';
  end
  nt = i-1;

end


% Integrate an ODE forward in time using Bogey & Bailly's RKo6s scheme (as used by Gwenael), with filtering each step
function [v, t_array, nt ] = bbo6s(xdot, ... % Function xdot(x, t) giving derivative (column vector)
				   filter, ... % Function filter(x, dt) filtering x
				   x, ... % Initial value of x (column vector)
				   t, ... % Initial time
				   dt, ... % Time increment
				   T, ... % Final time
				   err, ... % Error tolerance (not used)
				   varargin); % Any additional flags
	 
  % Sanity check : x should be a column vector
  if (size(x)(2) ~= 1)
    fprintf(stderr(), "bbo6s: Warning: x is not a column vector!\n");
    x = x(:);
  end

  % Flags:
  record = true; % Should we record each step?
  
  % Process additional flags
  for i = 1:length(varargin)
    switch (varargin{i})
      case "final"
	% Only return the final result, not the steps on the way
	record = false;
      otherwise
	fprintf(stderr(), "bbo6s: Warning: unknown flag \"%s\"\n", varargin{i});
    end
  end  

  % RKo6s coefficients
  g1 = 1;
  g2 = 1./2;
  g3 = 0.165919771368;
  g4 = 0.040919732041;
  g5 = 0.007555704391;
  g6 = 0.000891421261;

  % Conversion of coefficients
  a6 = g1;
  a5 = g2./g1;
  a4 = g3./g2;
  a3 = g4./g3;
  a2 = g5./g4;
  a1 = g6./g5;
  
  if (record)
    % Preallocate space for v and t_array
    v = zeros(ceil(T./dt+1), rows(x));
    t_array = zeros(ceil(T./dt+1), 1);
    v(1,:) = x.';
    t_array(1) = t;
  end
  i = 1;

  while (t < T)

    % Are we taking the last step?
    if (t + dt >= T)
       dt = T - t;
       T = 0.5.*t-1; % Ensures T < t, and so ensures this is the last iteration
    end

    % B&B's RKo6s
    lst = dt.*xdot(x          , t        ); % k1
    lst = dt.*xdot(x + a1.*lst, t + a1.*dt); % k2
    lst = dt.*xdot(x + a2.*lst, t + a2.*dt); % k3
    lst = dt.*xdot(x + a3.*lst, t + a3.*dt); % k4
    lst = dt.*xdot(x + a4.*lst, t + a4.*dt); % k5
    lst = dt.*xdot(x + a5.*lst, t + a5.*dt); % k6
    x = x + a6.*lst;
    x = filter(x, dt);
    t = t + dt;
    i = i + 1;
    if (record)
      v(i,:) = x.';
      t_array(i) = t;
    end

  end

  if (~record)
     % Record the final step if we haven't already
     t_array(1) = t;
     v(1,:) = x.';
  end
  nt = i-1;

end


% Integrate an ODE forward in time using Bogey & Bailly's RKo5s scheme, with filtering each step
function [v, t_array, nt ] = bbo5s(xdot, ... % Function xdot(x, t) giving derivative (column vector)
				   filter, ... % Function filter(x, dt) filtering x
				   x, ... % Initial value of x (column vector)
				   t, ... % Initial time
				   dt, ... % Time increment
				   T, ... % Final time
				   err, ... % Error tolerance (not used)
				   varargin); % Any additional flags
	 
  % Sanity check : x should be a column vector
  if (size(x)(2) ~= 1)
    fprintf(stderr(), "bbo5s: Warning: x is not a column vector!\n");
    x = x(:);
  end

  % Flags:
  record = true; % Should we record each step?
  
  % Process additional flags
  for i = 1:length(varargin)
    switch (varargin{i})
      case "final"
	% Only return the final result, not the steps on the way
	record = false;
      otherwise
	fprintf(stderr(), "bbo5s: Warning: unknown flag \"%s\"\n", varargin{i});
    end
  end  

  % RKo6s coefficients
  g1 = 1;
  g2 = 1./2;
  g3 = 0.165250353664;
  g4 = 0.039372585984;
  g5 = 0.007149096448;

  % Conversion of coefficients
  a5 = g1;
  a4 = g2./g1;
  a3 = g3./g2;
  a2 = g4./g3;
  a1 = g5./g4;
  
  if (record)
    % Preallocate space for v and t_array
    v = zeros(ceil(T./dt+1), rows(x));
    t_array = zeros(ceil(T./dt+1), 1);
    v(1,:) = x.';
    t_array(1) = t;
  end
  i = 1;

  while (t < T)

    % Are we taking the last step?
    if (t + dt >= T)
       dt = T - t;
       T = 0.5.*t-1; % Ensures T < t, and so ensures this is the last iteration
    end

    % B&B's RKo6s
    lst = dt.*xdot(x          , t         ); % k1
    lst = dt.*xdot(x + a1.*lst, t + a1.*dt); % k2
    lst = dt.*xdot(x + a2.*lst, t + a2.*dt); % k3
    lst = dt.*xdot(x + a3.*lst, t + a3.*dt); % k4
    lst = dt.*xdot(x + a4.*lst, t + a4.*dt); % k5
    x = x + a5.*lst;
    x = filter(x, dt);
    t = t + dt;
    i = i + 1;
    if (record)
      v(i,:) = x.';
      t_array(i) = t;
    end

  end

  if (~record)
     % Record the final step if we haven't already
     t_array(1) = t;
     v(1,:) = x.';
  end
  nt = i-1;
  
end

% Integrate an ODE forward in time using Bogey & Bailly's RKs4s scheme (allegedly a "standard" RK4 scheme), with filtering each step
function [v, t_array, nt ] = bbs4s(xdot, ... % Function xdot(x, t) giving derivative (column vector)
				   filter, ... % Function filter(x, dt) filtering x
				   x, ... % Initial value of x (column vector)
				   t, ... % Initial time
				   dt, ... % Time increment
				   T, ... % Final time
				   err, ... % Error tolerance (not used)
				   varargin); % Any additional flags
	 
  % Sanity check : x should be a column vector
  if (size(x)(2) ~= 1)
    fprintf(stderr(), "bbs4s: Warning: x is not a column vector!\n");
    x = x(:);
  end

  % Flags:
  record = true; % Should we record each step?
  
  % Process additional flags
  for i = 1:length(varargin)
    switch (varargin{i})
      case "final"
	% Only return the final result, not the steps on the way
	record = false;
      otherwise
	fprintf(stderr(), "bbs4s: Warning: unknown flag \"%s\"\n", varargin{i});
    end
  end  

  % RKo6s coefficients
  g1 = 1;
  g2 = 1./2;
  g3 = 1./6;
  g4 = 1./24;

  % Conversion of coefficients
  a4 = g1;
  a3 = g2./g1;
  a2 = g3./g2;
  a1 = g4./g3;
  
  if (record)
    % Preallocate space for v and t_array
    v = zeros(ceil(T./dt+1), rows(x));
    t_array = zeros(ceil(T./dt+1), 1);
    v(1,:) = x.';
    t_array(1) = t;
  end
  i = 1;

  while (t < T)

    % Are we taking the last step?
    if (t + dt >= T)
       dt = T - t;
       T = 0.5.*t-1; % Ensures T < t, and so ensures this is the last iteration
    end

    % B&B's RKo6s
    lst = dt.*xdot(x          , t         ); % k1
    lst = dt.*xdot(x + a1.*lst, t + a1.*dt); % k2
    lst = dt.*xdot(x + a2.*lst, t + a2.*dt); % k3
    lst = dt.*xdot(x + a3.*lst, t + a3.*dt); % k4
    x = x + a4.*lst;
    x = filter(x, dt);
    t = t + dt;
    i = i + 1;
    if (record)
      v(i,:) = x.';
      t_array(i) = t;
    end

  end

  if (~record)
     % Record the final step if we haven't already
     t_array(1) = t;
     v(1,:) = x.';
  end
  nt = i-1;

end
