% Copyright (c) 2016 Ed Brambley <E.J.Brmabley@damtp.cam.ac.uk>.
% All rights reserved.

0;

% 2D filter on x, returning a column vector
function fx = filter_2d(x, filter, strength)
  fx = (filter(reshape(x, 2, rows(x(:))./2).', strength).')(:);
end


% The null filter
function x = filter_null(x, strength)
end


% Filter every DT time units.
function x = filter_intermittent(x, dt, filter, strength, DT)

  persistent t = 0;

  t = t + dt;
  if (t < DT)
    return;
  end
  t = 0;

  x = filter(x, strength);
end


function fx = perform_periodic_filter(x, strength, a, w)

  % Preallocate space
  fx = x;
  N = rows(x);

  % Periodically extend x by w points on the left and right
  x = [x((N+1-w):N,:); x; x(1:w,:)];

  % Apply filter
  for j = 0:(2.*w)
    fx = fx - strength.*a(j+1).*x((1+j):(N+j),:);
  end
end


% S7: Standard 6th order periodic filtering independently on each column of x
function fx = filter_s7(x, strength)

  % Define filter
  a0 = +0.3125;
  a1 = -0.234375;
  a2 = +0.09375;
  a3 = -0.015625;
  a = [a3, a2, a1, a0, a1, a2, a3];
  w = 3;

  fx = perform_periodic_filter(x, strength, a, w);
end


% N7: 4th order at both 0 and 1 periodic filtering independently on each column of x
function fx = filter_n7(x, strength)

  % Define filter
  a0 = +0.5;
  a1 = -0.28125;
  a2 = +0.0;
  a3 = +0.03125;
  a = [a3, a2, a1, a0, a1, a2, a3];
  w = 3;

  fx = perform_periodic_filter(x, strength, a, w);
end


% S82: 8th order at 0 and 2nd order at pi
function fx = filter_s82(x, strength)

  % Define filter
  a0 = +70./256;
  a1 = -56./256;
  a2 = +28./256;
  a3 = - 8./256;
  a4 = + 1./256;

  a = [a4, a3, a2, a1, a0, a1, a2, a3, a4];
  w = 4;

  fx = perform_periodic_filter(x, strength, a, w);
end


% S84: 8th order at 0 and 4th order at pi
function fx = filter_s84(x, strength)

  % Define filter
  a0 = +98./256;
  a1 = -70./256;
  a2 = +20./256;
  a3 = + 5./256;
  a4 = - 5./256;
  a5 = + 1./256;

  a = [a5, a4, a3, a2, a1, a0, a1, a2, a3, a4, a5];
  w = 5;

  fx = perform_periodic_filter(x, strength, a, w);
end


% P11: 6th order at 0 and pinner to 1 at 2 pi/3 periodic filtering independently on each column of x
function fx = filter_p11(x, strength)

  % Define filter
  a0 = +241./432;
  a1 = -499./1728;
  a2 = -47./864;
  a3 = +47./864;
  a4 = +11./432;
  a5 = -1./64;
  a = [a5, a4, a3, a2, a1, a0, a1, a2, a3, a4, a5];
  w = 5;

  fx = perform_periodic_filter(x, strength, a, w);
end

% S104: 10th order at 0 and 4th order at pi
function fx = filter_s104(x, strength)

  % Define filter
  a0 =  357./1024;
  a1 = -135./512;
  a2 =  405./4096;
  a3 =    5./1024;
  a4 =  -45./2048;
  a5 =   36./4096;
  a6 =   -5./4096;

  a = [a6, a5, a4, a3, a2, a1, a0, a1, a2, a3, a4, a5, a6];
  w = 6;

  fx = perform_periodic_filter(x, strength, a, w);
end


% S164: 16th order at 0 and 4th order at pi
function fx = filter_s164(x, strength)

  % Define filter
  a = [ 18590, -15444, 8424, -2184, -756, 1080, -552, 162, -27, 2].*2.^(-16);
  w = length(a)-1;
  a = [ fliplr(a(2:end)) , a ];

  fx = perform_periodic_filter(x, strength, a, w);
end


% Bogey & Bailly's SFo9p (4th order)
function fx = filter_bbo9p(x, strength)

  % Define filter
  a0 = +0.243527493120;
  a1 = -0.204788880640;
  a2 = +0.120007591680;
  a3 = -0.045211119360;
  a4 = +0.008228661760;

  a = [a4, a3, a2, a1, a0, a1, a2, a3, a4];
  w = 4;

  fx = perform_periodic_filter(x, strength, a, w);
end

% Bogey & Bailly's SFo11p (2nd order)
function fx = filter_bbo11p(x, strength)

  % Define filter
  a0 = +0.215044884112;
  a1 = -0.187772883589;
  a2 = +0.123755948787;
  a3 = -0.059227575576;
  a4 = +0.018721609157;
  a5 = -0.002999540835;

  a = [a5, a4, a3, a2, a1, a0, a1, a2, a3, a4, a5];
  w = 5;

  fx = perform_periodic_filter(x, strength, a, w);
end

% Bogey & Bailly's SFo13p (4th order)
function fx = filter_bbo13p(x, strength)

  % Define filter
  a0 = +0.190899511506;
  a1 = -0.171503832236;
  a2 = +0.123632891797;
  a3 = -0.069975429105;
  a4 = +0.029662754736;
  a5 = -0.008520738659;
  a6 = +0.001254597714;

  a = [a6, a5, a4, a3, a2, a1, a0, a1, a2, a3, a4, a5, a6];
  w = 6;

  fx = perform_periodic_filter(x, strength, a, w);
end
