% Copyright (c) 2016 Ed Brambley <E.J.Brmabley@damtp.cam.ac.uk>.
% All rights reserved.

0;

function v = maximum_order(Lm, Lp, Nm, Np)
  % Calculate the coefficients \beta_j and d_n such that
  % \Sum_{j=-Lm}^Lp \beta_j f'_j = \frac{1}{\delta}\Sum_{j=-Nm}^Np d_j f_j
  % gives a maximumal order derivative with error O(\delta^{Lm+Lp+Nm+Np}),
  % where f_j = f(\delta j) and f'_j = f'(\delta j) and \beta_0=1.
  % v(1:(Lm+Lp+1)) contain \beta_j for j=-Lm:Lp
  % v((Lm+Lp+2):(Lm+Lp+Nm+Np+2) contain d_j for j=-Nm:Np

  M = Lm+Lp+Nm+Np;
  A=zeros(M+1,M+1);
  x=zeros(M+1,1);
  j = [ -Lm:-1 , 1:Lp ];
  m = -Nm:Np;
  for n = 0:M
    A(n+1, :) = [ -n.*j.^(n-1) , m.^n ];
    x(n+1) = (n == 1);
  end

  v = A \ x;
  v = [ v(1:Lm) ; 1.; v((Lm+1):(M+1)) ];
end
  

function [B, D] = derivative_matrices_2d_circular(N, ... % points are 0, ..., N, delta x = 1/(N+1)
						  Lm, Lp, Nm, Np, ...
						  T) % Type
  % Calculate derivative matrices for v = (p_0, xi_0, p_1, xi_1, ..., p_N, xi_N)
	
  switch (T)
    case "None" % Use maximal order
      v = maximum_order(Lm, Lp, Nm, Np);
      beta  = v(1:(Lm+Lp+1)    ).';
      delta = v(  (Lm+Lp+2):end).';

    case "MO5"
      beta = [ 12 ];
      delta = [1, -8, 0, 8, -1];
      Lm = Lp = 0;
      Nm = Np = 2;

    case "MO7"
      beta = [ 60 ];
      delta = [-1, 9, -45, 0, 45, -9, 1];
      Lm = Lp = 0;
      Nm = Np = 3;

    case "MO15"
      beta = [ 360360 ];
      delta = [-15, 245, -1911, 9555, -35035, 105105, -315315, 0, 315315, -105105, 35035, -9555, 1911, -245, 15];
      Lm = Lp = 0;
      Nm = Np = 7;

    case "DRP7" % Tam & Webb (1993) DRP scheme optimized over pi/2
      beta  = [ 1 ];
      delta = [ 0.79926643, 0, 0];
      delta(2) =  9./20. - delta(1)*4./5.;
      delta(3) = -2./15. + delta(1)   /5.;
      delta = [ -fliplr(delta) , 0, delta ];
      Lm = Lp = 0;
      Nm = Np = 3;

    case "TSDRP" % Tam & Shen (1993) optimized over 1.1
      beta  = [ 1 ];
      delta = [ 0.77088238051822552, 0, 0];
      delta(2) =  9./20. - delta(1)*4./5.;
      delta(3) = -2./15. + delta(1)   /5.;
      delta = [ -fliplr(delta) , 0, delta ];
      Lm = Lp = 0;
      Nm = Np = 3;

    case "DRP15" % Tam (2012) 15-point optimized over 1.8 (only 4th order)
      beta  = [ 1 ];
      delta = [  9.1942501110343045059277722885e-1, ...
	        -3.5582959926835268755667642401e-1, ...
	         1.5251501608406492469104928679e-1, ...
	        -5.9463040829715772666828596899e-2, ...
	         1.9010752709508298659849167988e-2, ...
	        -4.3808649297336481851137000907e-3, ...
	         5.3896121868623384659692955878e-4 ];
      delta = [ -fliplr(delta) , 0, delta ];
      Lm = Lp = 0;
      Nm = Np = 7;

				% TODO:
				% Tam (2012) 9-point optimized over 1.28
				% Tam (2012) 11-point optimized over 1.45
				% Tam (2012) 13-point optimized over 1.63

    case "KL2" % Kim & Lee (1996) tridiagonal 2nd order "maximum resolution"
      beta  = [ 1, 0.450901855 ];
      delta = [ 1.545790417/2., 0.434249728/4., -0.078236437/6. ];
      beta  = [  fliplr(beta ) ,    beta(2:end) ];
      delta = [ -fliplr(delta) , 0, delta       ];
      Lm = Lp = 1;
      Nm = Np = 3;

    case "KL4" % Kim & Lee (1996) tridiagonal 4th order "maximum resolution"
      beta  = [ 1, 0.435181352 ];
      delta = [ 1.551941906/2., 0.361328195/4., -0.042907397/6. ];
      beta  = [  fliplr(beta ) ,    beta(2:end) ];
      delta = [ -fliplr(delta) , 0, delta       ];
      Lm = Lp = 1;
      Nm = Np = 3;

    case "KL6" % Kim & Lee (1996) tridiagonal 6th order "maximum resolution"
      beta  = [ 1, 0.408589269 ];
      delta = [ 1.568098212/2., 0.271657107/4., -0.022576781/6. ];
      beta  = [  fliplr(beta ) ,    beta(2:end) ];
      delta = [ -fliplr(delta) , 0, delta       ];
      Lm = Lp = 1;
      Nm = Np = 3;

    case "KL8" % Kim & Lee (1996) tridiagonal 8th order  maximum order
      beta  = [ 480, 180 ];
      delta = [ 375, 24, -1 ];
      beta  = [  fliplr(beta ) ,    beta(2:end) ];
      delta = [ -fliplr(delta) , 0, delta       ];
      Lm = Lp = 1;
      Nm = Np = 3;

    case "TBf" % Tang & Baeder (1999) Fully trig
      beta  = [ 1 ];
      delta = [ 0.825899, -0.221299, 0.040501 ];
      delta = [ -fliplr(delta) , 0, delta ];
      Lm = Lp = 0;
      Nm = Np = 3;

    case "TB11" % Tang & Baeder (1999) Mixed 1st k=1
      beta  = [ 1 ];
      delta = [ 0.776932, -0.172333, 0.022578 ];
      delta = [ -fliplr(delta) , 0, delta ];
      Lm = Lp = 0;
      Nm = Np = 3;

    case "TB13" % Tang & Baeder (1999) Mixed 1st k=3
      beta  = [ 1 ];
      delta = [ 0.823497, -0.218898, 0.038099 ];
      delta = [ -fliplr(delta) , 0, delta ];
      Lm = Lp = 0;
      Nm = Np = 3;

    case "TB41" % Tang & Baeder (1999) Mixed 4th k=1
      K = 1;
      d = 12*sin(K*pi/2) - 48*sin(K*pi/3) + 60*sin(K*pi/6)
      beta  = [ 1 ];
      delta = [  (5*K*pi + 8*sin(K*pi/2) - 27*sin(K*pi/3))/d , ...
                -(4*K*pi +   sin(K*pi/2) - 27*sin(K*pi/6))/d , ...
	         (  K*pi +   sin(K*pi/3) -  8*sin(K*pi/6))/d ];
      delta = [ -fliplr(delta) , 0, delta ];
      Lm = Lp = 0;
      Nm = Np = 3;

    case "TB43" % Tang & Baeder (1999) Mixed 4th k=3
      K = 3;
      d = 12*sin(K*pi/2) - 48*sin(K*pi/3) + 60*sin(K*pi/6)
      beta  = [ 1 ];
      delta = [  (5*K*pi + 8*sin(K*pi/2) - 27*sin(K*pi/3))/d , ...
                -(4*K*pi +   sin(K*pi/2) - 27*sin(K*pi/6))/d , ...
	         (  K*pi +   sin(K*pi/3) -  8*sin(K*pi/6))/d ];
      delta = [ -fliplr(delta) , 0, delta ];
      Lm = Lp = 0;
      Nm = Np = 3;

    case "BBo9p" % Bogey & Bailly (2004) FDo9p
      beta = [ 1 ];
      delta = [ 0.841570125482, -0.244678631765, 0.059463584768, -0.007650904064 ];
      delta = [ -fliplr(delta) , 0, delta ];
      Lm = Lp = 0;
      Nm = Np = 4;

    case "BBo11p" % Bogey & Bailly (2004) FDo11p
      beta = [ 1 ];
      delta = [ 0.872756993962, -0.286511173973, 0.090320001280, -0.020779405824, 0.002484594688 ];
      delta = [ -fliplr(delta) , 0, delta ];
      Lm = Lp = 0;
      Nm = Np = 5;

    case "BBo13p" % Bogey & Bailly (2004) FDo13p
      beta = [ 1 ];
      delta = [ 0.907646591371, -0.337048393268, 0.133442885327, -0.045246480208, 0.011169294114, -0.001456501759 ];
      delta = [ -fliplr(delta) , 0, delta ];
      Lm = Lp = 0;
      Nm = Np = 6;

    otherwise
      fprintf(stderr(), "Unknown derivative type %s\n", T);
      return;
  end

  % Sanity check
  if (columns(beta) ~= Lm+Lp+1 || columns(delta) ~= Nm+Np+1)
    fprintf(stderr(), "derivative_matrices_2d_circular: Sanity check failed!");
    exit(1);
  end

  % Calculate derivative matrices such that Bv' = Dv/(delta X).
  B = spalloc(2*(N+1), 2*(N+1), 2*(N+1)*(Lm+Lp+1));
  D = spalloc(2*(N+1), 2*(N+1), 2*(N+1)*(Nm+Np+1));

  % Calculate the derivatives
  for i = 1:N+1
    if (i > Lm && i <= N+1-Lp) % Does this line fit?
      B(2*i-1, 2*i-1+2*((-Lm):Lp)) = beta;
      B(2*i  , 2*i  +2*((-Lm):Lp)) = beta;
    else
      for j = (-Lm):Lp % This line wraps, so do it point by point
	in = mod(i+j-1,N+1)+1; % Calculate wrapped index
	B(2*i-1, 2*in-1) = beta(j+Lm+1);
	B(2*i  , 2*in  ) = beta(j+Lm+1);
      end
    end
    if (i > Nm && i <= N+1-Np) % Does this line fit?
      D(2*i-1, 2*i-1+2*((-Nm):Np)) = delta;
      D(2*i  , 2*i  +2*((-Nm):Np)) = delta;
    else
      for j = (-Nm):Np % This line wraps, so do it point by point
	in = mod(i+j-1,N+1)+1; % Calculate wrapped index
	D(2*i-1, 2*in-1) = delta(j+Nm+1);
	  D(2*i  , 2*in  ) = delta(j+Nm+1);
      end
    end
  end
end
