% Copyright (c) 2016 Ed Brambley <E.J.Brmabley@damtp.cam.ac.uk>.
% All rights reserved.

0;

% Calculate a polynomial bump function with a discontinuous (bump_degree)th derivative

global bump_coeffs = [];
global bump_degree = 8; % While you would have thought 15th order
% would be needed to get convergence of the MO15, in fact degree 8
% seems to be the sweet spot balancing error within the bump with
% error at the edges, although DRP15 prefers 6 (as it would, being
% only 4th order accurate).

function [b, bp] = bump(y) % Assume y is a row vector

  global bump_coeffs;
  global bump_degree;

  % Shorthand
  n = bump_degree;
  a = bump_coeffs;

  % Preserve the shape of y, and convert it to a column vector
  shape = size(y);
  y = y(:);


  % Have we calculated the correct coefficients yet?
  if (rows(a) ~= n)

    % Calculate the coefficients a_j s.t. b(x) = x^n \sum_{j=0}^{n-1} a_j (1-x)^j is a bump function with b(0) = 0 and b(1)=1.
    % a_j = nchoosek(n+j-1, j).

    a = zeros(n,1);
    cum = 1;
    for r=1:n % r = j+1
      a(r) = cum;
      cum = cum .* (n+r-1) ./ r;
    end

    % Save as a global variable.  bump_degree is already correctly set
    bump_coeffs = a;
  end


  % Calculate \sum_{j=0}^{n-1} a_j (1-y)^j, used later to calculate both the bump function and (if requested) its derivative
  b = (repmat(1-y,1,n).^repmat(0:(n-1),rows(y),1))*a;

  % Calculate derivative if requested
  if ( nargout > 1 )

     % Calculate \sum_{j=0}^{n-1} -j a_j (1-y)^{j-1}
    bp = (repmat(-(0:(n-1)),rows(y),1).*repmat(1-y,1,n).^repmat((0:(n-1))-1,rows(y),1))*a;

    % Calculate d/dy(y^n b)
    bp = y.^n.*bp + n.*y.^(n-1).*b;

    % Clip
    bp(y<=0) = 0;
    bp(y>=1) = 0;
    
    % Remap to original shape
    bp = reshape(bp, shape);

  end

  % Finish calculating bump function: Calculate y^n \sum_{j=0}^{n-1} a_j (1-y)^j
  b = y.^n.*b;

  % Clip
  b(y<=0) = 0;
  b(y>=1) = 1;
  
  % Remap to original shape
  b = reshape(b, shape);

end

