/* Copyright (C) 2013 E.J. Brambley

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 3 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses>.

   Additional permission under GNU GPL version 3 section 7

   If you modify this Program, or any covered work, by linking or
   combining it with D.E. Amos' Algorithm 644
   (http://www.netlib.org/toms/644) (or a modified version of that
   library), containing parts covered by the terms of the ACM Software
   Copyright and License Agreement
   (www.acm.org/publications/policies/softwarecrnotice), the licensors
   of this Program grant you additional permission to convey the
   resulting work.  Corresponding Source for a non-source form of such
   a combination shall include the source code for the parts of
   Algorithm 644 used as well as that of the covered work.

   If you modify this Program, or any covered work, by linking or
   combining it with LAPACK (http://www.netlib.org/lapack) (or a
   modified version of that library), containing parts covered by the
   terms of the LAPACK modified BSD license
   (http://www.netlib.org/lapack/LICENSE.txt), the licensors of this
   Program grant you additional permission to convey the resulting
   work.
*/

/* 
   This code forms the supplementary material of the publication
   Brambley & Gabard (2014), Journal of Sound and Vibration
   Please acknowledge use of this code by citing that publication.
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <complex.h>
#include <time.h>
#ifdef _OPENMP
#include <omp.h>
#endif
#include "solve_for_surface_waves.h"


static inline double cabs2(const complex double x)
{
  return creal(x)*creal(x)+cimag(x)*cimag(x);
}

static inline complex double calc_alpha(const complex double k, const complex double omega, const double M)
{
  return -I*csqrt(I*(k + omega/(1.-M)))*csqrt(-I*(k - omega/(1.+M))*(1.-M*M));
}

static inline complex double calc_alpha_old(const complex double k, const complex double omega, const double M)
{
  return -I*csqrt(k*k - (omega-M*k)*(omega-M*k));
}


int main(int argc, char** argv)
{
  double accuracy, omegar, omegai, M, delta, Zr, Zi;
  complex double omega, Z;
  int verbose = 0;

  /*********/
  /* Setup */
  /*********/


  /* Check for command line switches */
  {
    int j = 0;
    for (int i = 0; i < argc; i++)
      {
	if (strcmp(argv[i], "-v"                 ) == 0 ||
	    strcmp(argv[i], "--verbose"          ) == 0   )
	  verbose = 1;
	else
	  argv[j++] = argv[i];
      }
    argc = j;
  }

  /* Input parameters */
  if (argc != 8 ||
      sscanf(argv[ 1], " %lg", &omegar         ) != 1 ||       omegar <= 0. ||
      sscanf(argv[ 2], " %lg", &omegai         ) != 1 ||
      sscanf(argv[ 3], " %lg", &M              ) != 1 ||            M < 0.  ||
      sscanf(argv[ 4], " %lg", &delta          ) != 1 ||        delta < 0.  ||
      sscanf(argv[ 5], " %lg", &accuracy       ) != 1 ||     accuracy < 0.  ||
      sscanf(argv[ 6], " %lg", &Zr             ) != 1 ||           Zr < 0.  ||
      sscanf(argv[ 7], " %lg", &Zi             ) != 1  )
    {
      fprintf(stderr, "\nBad command line!\n\nUsage: %s (options) <real omega> <imag omega> <M> <delta> <accuracy> <real Z> <imag Z>\n\n"
	      "Options:\n"
              "  -v  --verbose   Verbose mode\n"
              "\n\n", *argv);
      return 1;
    }

  /* Process input parameters */
  omega = omegar + I*omegai;
  Z = Zr + I*Zi;
  

  /****************************************/
  /* Calculate boundary layer thicknesses */
  /****************************************/

  /* Assume a linear, constant density boundary layer */
  double dmass = 0.;
  double dmom = 0.5*delta;
  double dke = 2.*delta/3.;
  double d1 = delta;


  /***************************/
  /* Calculate surface modes */
  /***************************/
  
  if (verbose)
    fprintf(stderr, "Calculating surface waves... ");

  /* Find poles */
  complex double modes[MAX_NUM_SURFACE_WAVES_MOD];
  int n = solve_for_surface_waves_bc(omega, modes, Z, M, dmass, dmom, dke, d1);

  if (verbose)
    fprintf(stderr, "done.\n");


  /**********/
  /* Output */
  /**********/

  if (verbose)
    fprintf(stderr, "Outputting... ");

  printf("# Surface_waves_modified, by E.J. Brambley\n"
	 "# \n"
	 "# omega = %g%+gi\n"
	 "# M = %g\n"
	 "# delta = %g\n"
	 "# accuracy = %g\n"
	 "# Z = %g%+gi\n"
	 "# \n\n",
	 creal(omega), cimag(omega), M, delta, accuracy, creal(Z), cimag(Z));

  /* Output modes: */
  printf("# type = (excited by a line source at some angles) + 2*(decays away from surface)\n");
  printf("# <real k> <imag k> <real alpha> <imag alpha> <type>\n");

  for (int i = 0; i < n; i++)
    {
      const complex double k = modes[i];
      const complex double Q = omega*omega*dmass - 2.*M*k*omega*dmom + k*k*M*M*dke;
      const complex double alpha = -((omega-M*k)*(omega-M*k) + I*M*Z*d1*k*k*k)/(omega*Z - I*Q);
      const int type = ( cabs2(1. - calc_alpha(k, omega, M)/alpha) < 1. ? 1 : 0) + 
	( cabs2(1. - calc_alpha_old(k, omega, M)/alpha) < 1. ? 2 : 0);

      printf("%24.16g %24.16g %24.16g %24.16g %4d\n",
	     creal(k), cimag(k), creal(alpha), cimag(alpha), type);
    }

  if (verbose)
    fprintf(stderr, "done.\n");

  return 0;
}

