/* Copyright (C) 2013 E.J. Brambley

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 3 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses>.

   Additional permission under GNU GPL version 3 section 7

   If you modify this Program, or any covered work, by linking or
   combining it with D.E. Amos' Algorithm 644
   (http://www.netlib.org/toms/644) (or a modified version of that
   library), containing parts covered by the terms of the ACM Software
   Copyright and License Agreement
   (www.acm.org/publications/policies/softwarecrnotice), the licensors
   of this Program grant you additional permission to convey the
   resulting work.  Corresponding Source for a non-source form of such
   a combination shall include the source code for the parts of
   Algorithm 644 used as well as that of the covered work.

   If you modify this Program, or any covered work, by linking or
   combining it with LAPACK (http://www.netlib.org/lapack) (or a
   modified version of that library), containing parts covered by the
   terms of the LAPACK modified BSD license
   (http://www.netlib.org/lapack/LICENSE.txt), the licensors of this
   Program grant you additional permission to convey the resulting
   work.
*/

/* 
   This code forms the supplementary material of the publication
   Brambley & Gabard (2014), Journal of Sound and Vibration
   Please acknowledge use of this code by citing that publication.
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <complex.h>
#include <time.h>
#ifdef _OPENMP
#include <omp.h>
#endif
#include "solve_for_surface_waves.h"

void zbesh_(double* ZR, double* ZI, double* FNU, int* KODE, int* M, int* N, double* CYR, double* CYI, int* NZ, int* IERR);

/********************/
/*                  */
/* Global Variables */
/*                  */
/********************/

static const complex double* g_Kp = 0;
static const complex double* g_Km = 0;
static const complex double* g_Ap = 0;
static const complex double* g_Am = 0;
static const complex double* g_alphap = 0;
static const complex double* g_alpham = 0;
static int g_Kp_n = 0;
static int g_Km_n = 0;
static int g_verbose = 0;

/* Upper bound for number of integration iterations */
#define MAX_ITS 1000000


/****************/
/*              */
/* Aux routines */
/*              */
/****************/

static inline double cabs2(const complex double x)
{
  return creal(x)*creal(x)+cimag(x)*cimag(x);
}

static inline complex double calc_alpha(const complex double k, const complex double omega, const double M)
{
  /* This is the optimized version */
  return -I*csqrt(I*(k + omega/(1.-M)))*csqrt(-I*(k - omega/(1.+M))*(1.-M*M));
}

static inline complex double srt(const complex double z)
{
  return csqrt(I*z)*((1.-I)/sqrt(2.));
}


/*************************************/
/* Calculate the free-field pressure */
/*************************************/

static inline complex double calc_pf(const double x,
				     const double y,
				     const complex double omega,
				     const double M)
{
  const double beta2 = 1.-M*M;
  const double beta = sqrt(beta2);
  const double r = sqrt(x*x + beta2*y*y);
  const complex double z = omega*r/beta2;

  /* Calculate h0^(2) and h0(2)' */
  int nz, ierr;
  int kode = 2; /* 1 for unscaled, 2 for scaled (cy*exp(-I*z) = H(2,0,z) */
  int n = 2; /* Number in the sequence */
  int m = 2; /* Type of bessel function */
  double fnu = 0.;  /* Kind of bessel function */
  double zr = creal(z);
  double zi = cimag(z);
  double cyr[2], cyi[2];

  /* Call the bessel function */
  zbesh_(&zr, &zi, &fnu, &kode, &m, &n, cyr, cyi, &nz, &ierr);

  /* Flag an error */
  if (ierr && ierr != 3)
    {
      if (g_verbose && x != 0. && y != 0.)
	fprintf(stderr, "calc_pf: Error for (x,y) = (%g, %g)\n", x, y);
      return 0.;
    }
 
  /* Flag a warning */
  if (ierr || nz)
    {
      if (g_verbose)
	fprintf(stderr, "calc_pf: Warning for (x,y) = (%g, %g)\n", x, y);
    }

  const complex double H  =   cyr[0] + I*cyi[0] ;
  const complex double Hp = -(cyr[1] + I*cyi[1]);

  /* Calculate and return pfree */
  return 0.25*omega/(beta2*beta)*(H - Hp*I*M*x/r)*cexp(I*omega*M*x/beta2 - I*z);
}


/**********************************/
/* Calculate the pressure p(x, y) */
/**********************************/


static inline complex double pr_integrand(const double t,
					  const double x,
					  const double y,
					  const double r,
					  const double r2,
					  const complex double omega,
					  const double M,
					  const complex double Z,
					  const double dmass,
					  const double dmom,
					  const double dke,
					  const double d1)
{
  const double q = t/(1.-t*t);
  const double dq = (1.+t*t)/((1.-t*t)*(1.-t*t));
  const complex double root = srt(I*omega*r + 0.25*(1.-M*M)*q*q);
  const complex double k = omega/(1.-M*M)*(x/r-M) - I*0.5*x*q*q/r2 + y*q*root/r2;
  const complex double alpha = omega*y/r - I*0.5*(1.-M*M)*y*q*q/r2 - x*q*root/r2;
  const complex double Q = omega*omega*dmass - 2.*M*omega*k*dmom + k*k*M*M*dke;

  // return (omega-M*k)*exp(-0.5*q*q)*dq/root * (alpha*omega*Z - (omega-M*k)*(omega-M*k))/(alpha*omega*Z + (omega-M*k)*(omega-M*k));
  return (omega-M*k)*exp(-0.5*q*q)*dq/root * (alpha*omega*Z - I*alpha*Q - (omega-M*k)*(omega-M*k) - I*M*Z*d1*k*k*k)/(alpha*omega*Z - I*alpha*Q + (omega-M*k)*(omega-M*k) + I*M*Z*d1*k*k*k);
}

static complex double p_contour(const complex double kp,
				const double x,
				const double y,
				const complex double omega,
				const double M,
				const double accuracy)
{
  const double r2 = x*x+(1.-M*M)*y*y;
  const double r = sqrt(r2);

  /* Find an interval for q that contains Re(kp) */
  double ql = 0.;
  complex double kl = omega/(1.-M*M)*(x/r-M);
  double qr = 0.;
  complex double kr = 0.;
  if (creal(kp) < creal(kl))
    {
      kr = kl;
      ql = -0.5;
      do
	{
	  ql *= 2.;
	  kl = omega/(1.-M*M)*(x/r-M) - I*0.5*x*ql*ql/r2 + y*ql*srt(I*omega*r + 0.25*(1.-M*M)*ql*ql)/r2;
	}
      while (creal(kp) < creal(kl));
    }
  else
    {
      qr = 0.5;
      do
	{
	  qr *= 2.;
	  kr = omega/(1.-M*M)*(x/r-M) - I*0.5*x*qr*qr/r2 + y*qr*srt(I*omega*r + 0.25*(1.-M*M)*qr*qr)/r2;
	}
      while (creal(kp) > creal(kr));
    }

  /* Subdivide interval until we are close to Re(kp) */
  while (creal(kr)-creal(kl) > 2.*accuracy || fabs(cimag(kr)-cimag(kl)) > 2.*accuracy)
    {
      const double q = 0.5*(ql+qr);
      const complex double k = omega/(1.-M*M)*(x/r-M) - I*0.5*x*q*q/r2 + y*q*srt(I*omega*r + 0.25*(1.-M*M)*q*q)/r2;
      if (creal(k) > creal(kp))
	{ kr = k; qr = q; }
      else
	{ kl = k; ql = q; }
    }
   
  return 0.5*(kl+kr);
}


static inline complex double calc_pr_integral(const double x,
					      double y,
					      const complex double omega,
					      const double M,
					      const complex double Z,
					      const double accuracy,
					      const double dmass,
					      const double dmom,
					      const double dke,
					      const double d1)
{
  if (y < 0.) y = -y;

  /* Find limits */
  const double rs = x*x+(1.-M*M)*y*y;
  const double r = sqrt(rs);
  const double qinc = pow(cabs2(omega)*rs/((1.-M*M)*(1.-M*M)), 0.25)/16.;
  int gaps = (int) (2./qinc + 0.5);
  if (gaps < 32) gaps = 32;
  if (gaps > MAX_ITS) gaps = MAX_ITS;
  const double a = -1.;
  const double b =  1.;
  const double fa = 0.;
  const double fb = 0.; 

  /* Setup the inital integral to the first pts points */
  double step = (b-a)/gaps;
  complex double l2 = pr_integrand(a+    step, x, y, r, rs, omega, M, Z, dmass, dmom, dke, d1);
  complex double l3 = pr_integrand(a+2.0*step, x, y, r, rs, omega, M, Z, dmass, dmom, dke, d1);
  complex double r2 = pr_integrand(b-    step, x, y, r, rs, omega, M, Z, dmass, dmom, dke, d1);
  complex double r3 = pr_integrand(b-2.0*step, x, y, r, rs, omega, M, Z, dmass, dmom, dke, d1);
  complex double ret = (9./24.)*(fa+fb) + (28./24.)*(l2+r2) + (23./24.)*(l3+r3);
  for (int i = 3; i < gaps - 2 ; i++)
    ret += pr_integrand(a + step * i, x, y, r, rs, omega, M, Z, dmass, dmom, dke, d1);
  ret *= step;

  complex double old_ret;
  do
    {
      /* Save the old value, for an accuracy check later */
      old_ret = ret;

      /* Double the number of points, so halve the step length multiplier already used so far and calculate the new end points */
      gaps *= 2;
      step = (b-a)/gaps;
      ret *= 0.5;
      ret += ((1./24.)*(l3+r3) - (5./24.)*(l2+r2))*step;
      l3 = l2;
      r3 = r2;
      l2 = pr_integrand(a+step, x, y, r, rs, omega, M, Z, dmass, dmom, dke, d1);
      r2 = pr_integrand(b-step, x, y, r, rs, omega, M, Z, dmass, dmom, dke, d1);
      ret += (28./24.)*(l2 + r2)*step;

      /* Calculate the new points - don't calculate the ones we've already done */
      complex double sum = 0.;
      for (int i = 3; i < gaps - 2; i += 2)
	sum += pr_integrand(a + step * i, x, y, r, rs, omega, M, Z, dmass, dmom, dke, d1);
      ret += sum * step;
    }
  while (cabs2(ret - old_ret) > accuracy*accuracy*(1. + cabs2(ret))
	 && gaps < MAX_ITS);

  if (cabs2(ret - old_ret) > accuracy*accuracy*(1. + cabs2(ret))
      || isnan(creal(ret)) || isnan(cimag(ret)))
    {
      if (g_verbose)
	fprintf(stderr, "Error: integral not converged for:\n  type = pr\n  (x,y) = (%g,%g)\n", x, y);
    }

  return ret*0.25/M_PI*I*cexp(I*omega*(M*x-r)/(1.-M*M));
}


static inline complex double calc_p(const double x,
				    const double y,
				    const complex double omega,
				    const double M,
				    const double y0,
				    const double accuracy,
				    const int impedance,
				    const complex double Z,
				    const double dmass,
				    const double dmom,
				    const double dke,
				    const double d1)
{
  if (!impedance)
    {
      /* Calculate reflected solution */
      return calc_pf(x, y-y0, omega, M) + calc_pf(x, y+y0, omega, M);
    }
  
  /* Calculate pole contributions */
  complex double sum = 0.;
  for (int j = 0; j < g_Kp_n; j++)
    if (cimag(p_contour(g_Kp[j], x, y+y0, omega, M, accuracy)) > cimag(g_Kp[j]))
      sum += 0.5*(omega-M*g_Kp[j])*g_Ap[j]/g_alphap[j]*cexp(-I*g_Kp[j]*x - I*g_alphap[j]*(y+y0));
  for (int j = 0; j < g_Km_n; j++)
    if (cimag(p_contour(g_Km[j], x, y+y0, omega, M, accuracy)) < cimag(g_Km[j]))
      sum += 0.5*(omega-M*g_Km[j])*g_Am[j]/g_alpham[j]*cexp(-I*g_Km[j]*x - I*g_alpham[j]*(y+y0));
  
  /* Add on the direct wave and the steepest descent reflected pressure integral */
  return sum + calc_pf(x, y-y0, omega, M) + calc_pr_integral(x, y+y0, omega, M, Z, accuracy, dmass, dmom, dke, d1);
}


/****************/
/*              */
/* Main Routine */
/*              */
/****************/

int main(int argc, char** argv)
{
  int Nx, Ny;
  double accuracy, omegar, omegai, ymax, xmax, y0, M, delta, Zr, Zi;
  complex double omega, Z;
  int impedance = 1;
  int instability = 0;
  complex double instability_pos = 0.;


  /*********/
  /* Setup */
  /*********/


  /* Check for command line switches */
  {
    int j = 0;
    for (int i = 0; i < argc; i++)
      {
	if (strcmp(argv[i], "-h"    ) == 0 ||
	    strcmp(argv[i], "--hard") == 0   )
	  impedance = 0;
	else if (strncmp(argv[i], "-i", 2) == 0)
	  {
	    const char* str = argv[i] + 2;
	    if (!*str) str = argv[++i];
	    if (sscanf(str, " %lg, %lg", &(__real__ instability_pos), &(__imag__ instability_pos)) != 2)
	      {
		fprintf(stderr, "\nIncorrect argument to -i: expected <real>, <imag>\n\n");
		return 1;
	      }
	    instability = 1;
	  }
	else if (strcmp(argv[i], "-v"                 ) == 0 ||
		 strcmp(argv[i], "--verbose"          ) == 0   )
	  g_verbose = 1;
	else
	  argv[j++] = argv[i];
      }
    argc = j;
  }

  /* Input parameters */
  if (argc != 13 ||
      sscanf(argv[ 1], " %lg", &omegar         ) != 1 ||       omegar <= 0. ||
      sscanf(argv[ 2], " %lg", &omegai         ) != 1 ||
      sscanf(argv[ 3], " %lg", &M              ) != 1 ||            M <  0. ||
      sscanf(argv[ 4], " %lg", &delta          ) != 1 ||        delta <  0. ||
      sscanf(argv[ 5], " %lg", &y0             ) != 1 ||           y0 <= 0. ||
      sscanf(argv[ 6], " %lg", &xmax           ) != 1 ||         xmax <= 0. ||
      sscanf(argv[ 7], " %lg", &ymax           ) != 1 ||         ymax <= 0. ||
      sscanf(argv[ 8], " %d" , &Nx             ) != 1 ||           Nx < 2   ||
      sscanf(argv[ 9], " %d" , &Ny             ) != 1 ||           Ny < 2   ||
      sscanf(argv[10], " %lg", &accuracy       ) != 1 ||     accuracy < 0.  ||
      sscanf(argv[11], " %lg", &Zr             ) != 1 ||           Zr < 0.  ||
      sscanf(argv[12], " %lg", &Zi             ) != 1
      )
    {
      fprintf(stderr, "\nBad command line!\n\nUsage: %s (options) <real omega> <imag omega> <M> <delta> <y0> <xmax> <ymax> <Nx> <Ny> <accuracy> <real Z> <imag Z>\n\n"
	      "Options:\n"
	      "  -h  --hard      Ignore impedance and use a hard wall boundary condition\n"
	      "  -i<r,i>         Treat the mode nearest r + ii as an instability\n"
              "  -v  --verbose   Verbose mode\n"
              "\n", *argv);
      return 1;
    }

  /* Process input parameters */
  omega = omegar + I*omegai;
  Z = Zr + I*Zi;
  

  /****************************************/
  /* Calculate boundary layer thicknesses */
  /****************************************/

  /* Assume a linear, constant density boundary layer */
  double dmass = 0.;
  double dmom = 0.5*delta;
  double dke = 2.*delta/3.;
  double d1 = delta;


  /***************************/
  /* Calculate surface modes */
  /***************************/
  
  /* Find poles */
  complex double Kp[MAX_NUM_SURFACE_WAVES_MOD];
  complex double Km[MAX_NUM_SURFACE_WAVES_MOD];
  int max_K = solve_for_surface_waves_bc(omega, Kp, Z, M, dmass, dmom, dke, d1);
  
  /* Filter poles based on alpha */
  {
    const int max = max_K;
    max_K = 0;
    for (int i = 0; i < max; i++)
      {
	const complex double k = Kp[i];
	const complex double Q = omega*omega*dmass - 2.*M*k*omega*dmom + k*k*M*M*dke;
	const complex double alpha = -((omega-M*k)*(omega-M*k) + I*M*Z*d1*k*k*k)/(omega*Z - I*Q);

	/* Only include poles which are in the correct alpha plane */
	if (cabs2(1. - calc_alpha(k, omega, M)/alpha) < 1.)
	  Kp[max_K++] = k;
      }
  }

  /* Find closest pole to the given instability position */
  int instability_closest = 0;
  if (instability)
    {
      if (max_K > 0)
	{
	  double dist2 = cabs2(Kp[0] - instability_pos);
	  for (int i = 1; i < max_K; i++)
	    {
	      const double this_dist2 = cabs2(Kp[i] - instability_pos);
	      if (this_dist2 < dist2)
		{
		  dist2 = this_dist2;
		  instability_closest = i;
		}
	    }
	  if (g_verbose)
	    fprintf(stderr, "Found instability pole at %g%+gi (distance %g from given location).\n",
		    creal(Kp[instability_closest]), cimag(Kp[instability_closest]), sqrt(dist2));
	}
      else
	{
	  instability = 0;
	  if (g_verbose)
	    fprintf(stderr, "No poles found; ignoring request for instability pole.\n");
	}
    }

  /* Sort poles into K_+ and K_- and calculate coefficients */
  int Kp_n = 0;
  int Km_n = 0;
  complex double Ap[max_K];
  complex double Am[max_K];
  complex double alphap[max_K];
  complex double alpham[max_K];
  complex double instability_k = 0.;
  complex double instability_A = 0.;
  complex double instability_alpha = 0.;
  for (int i = 0; i < max_K; i++)
    {
      const complex double k = Kp[i];
      const complex double Q = omega*omega*dmass - 2.*omega*M*k*dmom + k*k*M*M*dke;
      const complex double Qk = 2.*M*(M*k*dke - omega*dmom);
      const complex double alpha = -((omega-M*k)*(omega-M*k) + I*M*Z*d1*k*k*k)/(omega*Z - I*Q);
      const complex double A = 2.*I*((omega-M*k)*(omega-M*k) + I*M*Z*d1*k*k*k)*alpha/(((1.-M*M)*k+M*omega)*(omega*Z - I*Q) + 2.*M*(omega-M*k)*alpha + I*alpha*alpha*Qk - 3.*I*M*Z*d1*k*k*alpha);

      if ((instability && i == instability_closest))
	{
	  instability_k     =  k;
	  instability_A     = -A;
	  instability_alpha = alpha;
	}

      if (cimag(k) < 0.)
	{
	  Km[Km_n] =  k;
	  Am[Km_n] = -A;
	  alpham[Km_n] = alpha;
	  Km_n++;
	}
      else
	{
	  Kp[Kp_n] = k;
	  Ap[Kp_n] = A;
	  alphap[Kp_n] = alpha;
	  Kp_n++;
	}
    }


  /* Save surface modes into global variables */
  g_Kp = Kp;
  g_Km = Km;
  g_Ap = Ap;
  g_Am = Am;
  g_alpham = alpham;
  g_alphap = alphap;
  g_Kp_n = Kp_n;
  g_Km_n = Km_n;


  /**********************/
  /* Calculate pressure */
  /**********************/

  /* Allocate memory */
  const int points = (2*Nx+1)*(Ny+1);
  complex double* const p = malloc(sizeof(complex double) * points);
  if (!p) { perror(*argv); return 2; }

  int processed = 0;
  time_t current_time = time(NULL)-1;
#ifdef _OPENMP
#pragma omp parallel for
#endif
  for (int i = 0; i < points; i++)
    {
      /* Progress report */
      if (g_verbose
#ifdef _OPENMP
	  && omp_get_thread_num() == 0
#endif
	  )
	{
	  time_t new_time = time(NULL);
	  if (new_time != current_time)
	    {
	      current_time = new_time;
	      fprintf(stderr, "%12d / %12d\r", processed, points);
	    }
	}

      const int iy = i/(2*Nx+1);
      const int ix = i%(2*Nx+1)-Nx;
      const double x = ix*xmax/Nx;
      const double y = iy*ymax/Ny;
      p[i] = calc_p(x, y, omega, M, y0, accuracy, impedance, Z, dmass, dmom, dke, d1);
      
#ifdef _OPENMP
#pragma omp atomic
#endif
      processed++;
    }

  if (g_verbose)
    fprintf(stderr, "\r%12d / %12d\n", processed, points);
      

  /**********/
  /* Output */
  /**********/

  /* Output header */
  printf("# Solver_modified, by E.J. Brambley\n"
	 "# \n"
	 "# omega = %g%+gi\n"
	 "# M = %g\n"
	 "# delta = %g\n"
	 "# y0 = %g\n"
	 "# (xmax, ymax) = (%g, %g)\n"
	 "# (Nx, Ny) = (%d, %d)\n"
	 "# accuracy = %g\n",
	 creal(omega), cimag(omega), M, delta, y0, xmax, ymax, Nx, Ny, accuracy);
  if (impedance)
    printf("# Z = %g%+gi\n", creal(Z), cimag(Z));
  else
    printf("# Z = infinite\n");
  if (instability)
    printf("# Instability at k = %g%+gi (searched near %g%+gi)\n",
	   creal(instability_k  ), cimag(instability_k  ),
	   creal(instability_pos), cimag(instability_pos));
  printf("# \n\n");
  

  /* Output pressure */
  printf("# <x> <y> <real p> <imag p>");
  if (instability)
    printf("<real p - instability> <imag p - instability>");
  printf("\n");
  for (int j = -Nx; j <= Nx; j++)
    {
      const double x = j*xmax/Nx;
      for (int i = 0; i <= Ny; i++)
	{
	  const double y = i*ymax/Ny;
	  const complex double P = p[i*(2*Nx+1)+j+Nx];
	  if (instability)
	    {
	      const complex double ic = 0.5*(omega-M*instability_k)*instability_A/instability_alpha*cexp(-I*instability_k*x - I*instability_alpha*(y+y0));
	      printf("%24.16g %24.16g %24.16g %24.16g %24.16g %24.16g\n", x, y, creal(P+ic), cimag(P+ic), creal(P), cimag(P));
	    }
	  else
	    {
	      printf("%24.16g %24.16g %24.16g %24.16g\n", x, y, creal(P), cimag(P));
	    }
	}
      printf("\n");
    }
  printf("\n");

      
  free(p);

  return 0;
}

