/* Copyright (C) 2013 E.J. Brambley

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 3 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses>.

   Additional permission under GNU GPL version 3 section 7

   If you modify this Program, or any covered work, by linking or
   combining it with D.E. Amos' Algorithm 644
   (http://www.netlib.org/toms/644) (or a modified version of that
   library), containing parts covered by the terms of the ACM Software
   Copyright and License Agreement
   (www.acm.org/publications/policies/softwarecrnotice), the licensors
   of this Program grant you additional permission to convey the
   resulting work.  Corresponding Source for a non-source form of such
   a combination shall include the source code for the parts of
   Algorithm 644 used as well as that of the covered work.

   If you modify this Program, or any covered work, by linking or
   combining it with LAPACK (http://www.netlib.org/lapack) (or a
   modified version of that library), containing parts covered by the
   terms of the LAPACK modified BSD license
   (http://www.netlib.org/lapack/LICENSE.txt), the licensors of this
   Program grant you additional permission to convey the resulting
   work.
*/

/* 
   This code forms the supplementary material of the publication
   Brambley & Gabard (2014), Journal of Sound and Vibration
   Please acknowledge use of this code by citing that publication.
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <complex.h>
#include <time.h>
#ifdef _OPENMP
#include <omp.h>
#endif
#include "solve_for_surface_waves.h"

void zbesh_(double* ZR, double* ZI, double* FNU, int* KODE, int* M, int* N, double* CYR, double* CYI, int* NZ, int* IERR);

/********************/
/*                  */
/* Global Variables */
/*                  */
/********************/

static const complex double* g_Kp = 0;
static const complex double* g_Km = 0;
static const complex double* g_Ap = 0;
static const complex double* g_Am = 0;
static const complex double* g_alphap = 0;
static const complex double* g_alpham = 0;
static int g_Kp_n = 0;
static int g_Km_n = 0;
static int g_verbose = 0;

/* Upper bound for number of integration iterations */
#define MAX_ITS 1000000



/****************/
/*              */
/* Aux routines */
/*              */
/****************/

static inline double cabs2(const complex double x)
{
  return creal(x)*creal(x)+cimag(x)*cimag(x);
}


static inline complex double calc_alpha(const complex double k, const complex double omega, const double M)
{
  return -I*csqrt(I*(k + omega/(1.-M)))*csqrt(-I*(k - omega/(1.+M))*(1.-M*M));
}

static inline complex double srt(const complex double z)
{
  return csqrt(I*z)*((1.-I)/sqrt(2.));
}


/****************************************************/
/* Calculate integral using I0 and steepest descent */
/****************************************************/

static inline complex double I0sd_integrand(const double q,
					    const double x,
					    const int dir,
					    const complex double omega,
					    const double M,
					    const complex double Z)
{
  const double t = q/(1.-q*q);
  const double dt = (1.+q*q)/((1.-q*q)*(1.-q*q));
  const complex double root = srt(1. - 0.25*I*t*t*(1.-M*M)/(x*dir*omega));
  /* const complex double k = omega/(1.-M*M)*(dir-M) - 0.5*I*t*t/x; */

  return t*t*exp(-0.5*t*t)*dt/(cexp(0.25*I*M_PI)*t + csqrt(dir*x*omega)/(Z*(1.-M*M)*(1.-M*M)*root)*(1.-dir*M*(1.-I*t*t*(1.-M*M)/(2.*omega*dir*x)))*(1.-dir*M*(1.-I*t*t*(1.-M*M)/(2.*omega*dir*x))));
}


static inline complex double calc_I0sd(const double x,
				       const int dir,
				       const complex double omega,
				       const double M,
				       const complex double Z,
				       const double accuracy)
{
  /* Find limits */
  const double qinc = pow(cabs2(0.25*x*omega/(1.-M*M)), 0.25);
  int gaps = (int) (2./qinc + 0.5);
  if (gaps < 8) gaps = 8;
  if (gaps > MAX_ITS) gaps = MAX_ITS;
  const double a = -1.;
  const double b =  1.;
  const double fa = 0.;
  const double fb = 0.; 

  /* Setup the inital integral to the first pts points */
  double step = (b-a)/gaps;
  complex double l2 = I0sd_integrand(a+    step, x, dir, omega, M, Z);
  complex double l3 = I0sd_integrand(a+2.0*step, x, dir, omega, M, Z);
  complex double r2 = I0sd_integrand(b-    step, x, dir, omega, M, Z);
  complex double r3 = I0sd_integrand(b-2.0*step, x, dir, omega, M, Z);
  complex double ret = (9./24.)*(fa+fb) + (28./24.)*(l2+r2) + (23./24.)*(l3+r3);
  for (int i = 3; i < gaps - 2 ; i++)
    ret += I0sd_integrand(a + step * i, x, dir, omega, M, Z);
  ret *= step;

  complex double old_ret;
  do
    {
      /* Save the old value, for an accuracy check later */
      old_ret = ret;

      /* Double the number of points, so halve the step length multiplier already used so far and calculate the new end points */
      gaps *= 2;
      step = (b-a)/gaps;
      ret *= 0.5;
      ret += ((1./24.)*(l3+r3) - (5./24.)*(l2+r2))*step;
      l3 = l2;
      r3 = r2;
      l2 = I0sd_integrand(a+step, x, dir, omega, M, Z);
      r2 = I0sd_integrand(b-step, x, dir, omega, M, Z);
      ret += (28./24.)*(l2 + r2)*step;

      /* Calculate the new points - don't calculate the ones we've already done */
      complex double sum = 0.;
      for (int i = 3; i < gaps - 2; i += 2)
	sum += I0sd_integrand(a + step * i, x, dir, omega, M, Z);
      ret += sum * step;
    }
  while (cabs2(ret - old_ret) > accuracy*accuracy*(1. + cabs2(ret))
	 && gaps < MAX_ITS);

  if (cabs2(ret - old_ret) > accuracy*accuracy*(1. + cabs2(ret))
      || isnan(creal(ret)) || isnan(cimag(ret)))
    {
      if (g_verbose && (g_verbose > 1 || x != 0.))
	fprintf(stderr, "Error: integral not converged for:\n  type = I0sd\n  x = %g\n  dir = %d\n", x, dir);
    }

  return ret*I/(M_PI*dir*x)*cexp(-I*omega*(dir-M)*x/(1.-M*M) + 0.25*I*M_PI);
}


/* Calculate the convolution function f(x) */
static inline complex double calc_I0(const double x,
				     const int dir,
				     const complex double omega,
				     const double M,
				     const complex double Z,
				     const double accuracy)
{
  complex double sum = 0.;

  if (dir < 0)
    for (int j = 0; j < g_Kp_n; j++)
      sum += g_Ap[j]*cexp(-I*g_Kp[j]*x);
  else
    for (int j = 0; j < g_Km_n; j++)
      sum += g_Am[j]*cexp(-I*g_Km[j]*x);

  return sum + calc_I0sd(x, dir, omega, M, Z, accuracy);
}


/****************************************************/
/* Calculate integral using I1 and steepest descent */
/****************************************************/

static inline complex double I1sd_integrand(const double q,
					    const double x,
					    const int dir,
					    const complex double omega,
					    const double M,
					    const complex double Z)
{
  const double t = q/(1.-q*q);
  const double dt = (1.+q*q)/((1.-q*q)*(1.-q*q));
  const complex double root = srt(1. - 0.25*I*t*t*(1.-M*M)/(x*dir*omega));
  /* const complex double k = omega/(1.-M*M)*(dir-M) - 0.5*I*t*t/x; */

  return Z*((dir-M)*(dir-M) + I*t*t*M*(1.-M*M)/(omega*x) - (1.-M*M)*(1.-M*M)*Z*t*root/srt(-I*omega*dir*x))/((1.-0.25*I*t*t*(1.-M*M)/(x*dir*omega))*-1.*Z*(1.-M*M)*(1.-M*M)*t/srt(-I*omega*dir*x) + root*(dir-M+0.5*I*t*t*(1.-M*M)*M/(omega*dir*x))*(dir-M+0.5*I*t*t*(1.-M*M)*M/(omega*dir*x)))*exp(-0.5*t*t)*dt;
}


static inline complex double calc_I1sd(const double x,
				       const int dir,
				       const complex double omega,
				       const double M,
				       const complex double Z,
				       const double accuracy)
{
  /* Find limits */
  const double qinc = pow(cabs2(0.25*x*omega/(1.-M*M)), 0.25);
  int gaps = (int) (2./qinc + 0.5);
  if (gaps < 8) gaps = 8;
  if (gaps > MAX_ITS) gaps = MAX_ITS;
  const double a = -1.;
  const double b =  1.;
  const double fa = 0.;
  const double fb = 0.; 

  /* Setup the inital integral to the first pts points */
  double step = (b-a)/gaps;
  complex double l2 = I1sd_integrand(a+    step, x, dir, omega, M, Z);
  complex double l3 = I1sd_integrand(a+2.0*step, x, dir, omega, M, Z);
  complex double r2 = I1sd_integrand(b-    step, x, dir, omega, M, Z);
  complex double r3 = I1sd_integrand(b-2.0*step, x, dir, omega, M, Z);
  complex double ret = (9./24.)*(fa+fb) + (28./24.)*(l2+r2) + (23./24.)*(l3+r3);
  for (int i = 3; i < gaps - 2 ; i++)
    ret += I1sd_integrand(a + step * i, x, dir, omega, M, Z);
  ret *= step;

  complex double old_ret;
  do
    {
      /* Save the old value, for an accuracy check later */
      old_ret = ret;

      /* Double the number of points, so halve the step length multiplier already used so far and calculate the new end points */
      gaps *= 2;
      step = (b-a)/gaps;
      ret *= 0.5;
      ret += ((1./24.)*(l3+r3) - (5./24.)*(l2+r2))*step;
      l3 = l2;
      r3 = r2;
      l2 = I1sd_integrand(a+step, x, dir, omega, M, Z);
      r2 = I1sd_integrand(b-step, x, dir, omega, M, Z);
      ret += (28./24.)*(l2 + r2)*step;

      /* Calculate the new points - don't calculate the ones we've already done */
      complex double sum = 0.;
      for (int i = 3; i < gaps - 2; i += 2)
	sum += I1sd_integrand(a + step * i, x, dir, omega, M, Z);
      ret += sum * step;
    }
  while (cabs2(ret - old_ret) > accuracy*accuracy*(1. + cabs2(ret))
	 && gaps < MAX_ITS);

  if (cabs2(ret - old_ret) > accuracy*accuracy*(1. + cabs2(ret))
      || isnan(creal(ret)) || isnan(cimag(ret)))
    {
      if (g_verbose && (g_verbose > 1 || x != 0.))
	fprintf(stderr, "Error: integral not converged for:\n  type = I1sd\n  x = %g\n  dir = %d\n", x, dir);
    }

  return ret*(1.-M*M)/(M_PI*M*M)*csqrt(omega/(x*dir))*cexp(-I*omega*(dir-M)*x/(1.-M*M)+I*0.25*M_PI);
}


/* Calculate the I1 steepest descent integral (including pole contributions, but excluding the Hankel function) */
static inline complex double calc_I1(const double x,
				     const int dir,
				     const complex double omega,
				     const double M,
				     const complex double Z,
				     const double accuracy)
{
  complex double sum = 0.;
  if (dir < 0)
    for (int j = 0; j < g_Kp_n; j++)
      sum += g_Ap[j]*cexp(-I*g_Kp[j]*x);
  else
    for (int j = 0; j < g_Km_n; j++)
      sum += g_Am[j]*cexp(-I*g_Km[j]*x);

  return sum + calc_I1sd(x, dir, omega, M, Z, accuracy);
}


static inline complex double calc_H(const double x,
				    const int dir,
				    const complex double omega,
				    const double M,
				    const complex double Z)
{
  int nz, ierr;
  int kode = 2; /* 1 for unscaled, 2 for scaled (cy = exp(-Q*z*I)*H(m,fnu,z), where Q=3-2*m) */
  int n = 1; /* Number in the sequence */
  int m = 2; /* Type of bessel function */
  double fnu = 0.;  /* Kind of bessel function */
  double zr = creal(omega*dir*x/(1.-M*M));
  double zi = cimag(omega*dir*x/(1.-M*M));
  double cyr, cyi;

  /* Call the bessel function */
  zbesh_(&zr, &zi, &fnu, &kode, &m, &n, &cyr, &cyi, &nz, &ierr);

  /* Flag an error */
  if (ierr && ierr != 3)
    {
      if (g_verbose && (g_verbose > 1 || zr*zr+zi*zi != 0.))
	fprintf(stderr, "zbesh: Error for z = %g%+gi\n", zr, zi);
      return NAN + I*NAN;
    }
 
  /* Flag a warning */
  if (ierr || nz)
    {
      if (g_verbose && (g_verbose > 1 || zr*zr+zi*zi != 0.))
	fprintf(stderr, "zbesh: Warning for z = %g%+gi\n", zr, zi);
    }

  return -sqrt(1.-M*M)*omega*Z*(cyr+I*cyi)*cexp(-I*omega*x*(dir-M)/(1.-M*M))/(M*M);
}


/****************/
/*              */
/* Main Routine */
/*              */
/****************/

int main(int argc, char** argv)
{
  int Nx;
  double accuracy, omegar, omegai, xmax, M, Zr, Zi;
  complex double omega, Z;
  int instability = 0;
  complex double instability_pos = 0.;


  /*********/
  /* Setup */
  /*********/


  /* Check for command line switches */
  {
    int j = 0;
    for (int i = 0; i < argc; i++)
      {
	if (strncmp(argv[i], "-i", 2) == 0)
	  {
	    const char* str = argv[i] + 2;
	    if (!*str) str = argv[++i];
	    if (sscanf(str, " %lg, %lg", &(__real__ instability_pos), &(__imag__ instability_pos)) != 2)
	      {
		fprintf(stderr, "\nIncorrect argument to -i: expected <real>, <imag>\n\n");
		return 1;
	      }
	    instability = 1;
	  }
	else if (strcmp(argv[i], "-v"                 ) == 0 ||
		 strcmp(argv[i], "--verbose"          ) == 0   )
	  g_verbose = 1;
	else if (strcmp(argv[i], "-V"                 ) == 0 ||
		 strcmp(argv[i], "--Verbose"          ) == 0   )
	  g_verbose = 2;
	else
	  argv[j++] = argv[i];
      }
    argc = j;
  }

  /* Input parameters */
  if (argc != 9 ||
      sscanf(argv[1], " %lg", &omegar         ) != 1 ||       omegar <= 0. ||
      sscanf(argv[2], " %lg", &omegai         ) != 1 ||
      sscanf(argv[3], " %lg", &M              ) != 1 ||            M <  0. ||
      sscanf(argv[4], " %lg", &xmax           ) != 1 ||         xmax <= 0. ||
      sscanf(argv[5], " %d" , &Nx             ) != 1 ||           Nx < 2   ||
      sscanf(argv[6], " %lg", &accuracy       ) != 1 ||     accuracy < 0.  ||
      sscanf(argv[7], " %lg" , &Zr            ) != 1 ||           Zr < 0.  ||
      sscanf(argv[8], " %lg" , &Zi            ) != 1   )
    {
      fprintf(stderr, "\nBad command line!\n\nUsage: %s (options) <real omega> <imag omega> <M> <xmax> <Nx> <accuracy> <real Z> <imag Z>\n\n"
	      "Options:\n"
	      "  -i<r,i>         Treat the mode nearest r + ii as an instability\n"
              "  -v  --verbose   Verbose mode\n"
              "  -V  --Verbose   Verbose mode (even for the integrals)\n"
              "\n", *argv);
      return 1;
    }

  /* Process input parameters */
  omega = omegar + I*omegai;
  Z = Zr + I*Zi;
  

  /***************************/
  /* Calculate surface modes */
  /***************************/
  
  /* Find poles */
  complex double Kp[MAX_NUM_SURFACE_WAVES];
  complex double Km[MAX_NUM_SURFACE_WAVES];
  int max_K = solve_for_surface_waves(omega, Kp, Z, M);
 
  /* Filter poles based on Im(alpha) */
  {
    const int max = max_K;
    max_K = 0;
    for (int i = 0; i < max; i++)
      {
	const complex double k = Kp[i];
	const complex double alpha = -(omega-M*k)*(omega-M*k)/(omega*Z);
	if (cabs2(1. - calc_alpha(k, omega, M)/alpha) < 1.)
	  Kp[max_K++] = k;
      }
  }

  /* Find closest pole to the given instability position */
  int instability_closest = 0;
  if (instability)
    {
      if (max_K > 0)
	{
	  double dist2 = cabs2(Kp[0] - instability_pos);
	  for (int i = 1; i < max_K; i++)
	    {
	      const double this_dist2 = cabs2(Kp[i] - instability_pos);
	      if (this_dist2 < dist2)
		{
		  dist2 = this_dist2;
		  instability_closest = i;
		}
	    }
	  if (g_verbose)
	    fprintf(stderr, "Found instability pole at %g%+gi (distance %g from given location).\n",
		    creal(Kp[instability_closest]), cimag(Kp[instability_closest]), sqrt(dist2));
	}
      else
	{
	  instability = 0;
	  if (g_verbose)
	    fprintf(stderr, "No poles found; ignoring request for instability pole.\n");
	}
    }

  /* Sort poles into K_+ and K_- and calculate coefficients */
  int Kp_n = 0;
  int Km_n = 0;
  complex double Ap[max_K];
  complex double Am[max_K];
  complex double alphap[max_K];
  complex double alpham[max_K];
  complex double instability_k = 0.;
  complex double instability_A = 0.;
  for (int i = 0; i < max_K; i++)
    {
      const complex double k = Kp[i];
      const complex double alpha = -(omega-M*k)*(omega-M*k)/(omega*Z);
      const complex double A = 2.*I*(omega-M*k)*(omega-M*k)*alpha/(omega*Z*((1.-M*M)*k+M*omega) +2.*M*(omega-M*k)*alpha);

      if ((instability && i == instability_closest))
	{
	  instability_k     =  k;
	  instability_A     = -A;
	}

      if (cimag(k) < 0.)
	{
	  Km[Km_n] =  k;
	  Am[Km_n] = -A;
	  alpham[Km_n] = alpha;
	  Km_n++;
	}
      else
	{
	  Kp[Kp_n] = k;
	  Ap[Kp_n] = A;
	  alphap[Kp_n] = alpha;
	  Kp_n++;
	}
    }


  /* Save surface modes into global variables */
  g_Kp = Kp;
  g_Km = Km;
  g_Ap = Ap;
  g_Am = Am;
  g_alpham = alpham;
  g_alphap = alphap;
  g_Kp_n = Kp_n;
  g_Km_n = Km_n;



  /**********/
  /* Output */
  /**********/

  /* Output header */
  printf("# Reflection, by E.J. Brambley\n"
	 "# \n"
	 "# omega = %g%+gi\n"
	 "# M = %g\n"
	 "# xmax = %g\n"
	 "# Nx = %d\n"
	 "# accuracy = %g\n",
	 creal(omega), cimag(omega), M, xmax, Nx, accuracy);
  printf("# Z = %g%+gi\n", creal(Z), cimag(Z));
  if (instability)
    printf("# Instability at k = %g%+gi (searched near %g%+gi)\n",
	   creal(instability_k  ), cimag(instability_k  ),
	   creal(instability_pos), cimag(instability_pos));
  printf("# \n\n");
  

  printf("# <x> <real I0> <imag I0> <real I1> <imag I1> <real H_0^(2)> <imag H_0^(2)>\n");
  int dir = -1;
  for (int i = -Nx; i <= Nx; i++)
    {
      const double x = i*xmax/Nx;
      const complex double ic = instability_A*cexp(-I*instability_k*x);
      const complex double I0 = calc_I0(x, dir, omega, M, Z, accuracy);
      const complex double I1 = calc_I1(x, dir, omega, M, Z, accuracy);
      const complex double H  = calc_H (x, dir, omega, M, Z);
      printf("%24.16g %24.16g %24.16g %24.16g %24.16g %24.16g %24.16g\n", x, creal(I0+ic), cimag(I0+ic), creal(I1+ic), cimag(I1+ic), creal(H), cimag(H));

      if (i == 0 && dir < 0)
	{
	  i--;
	  dir = 1;
	}
    }

  return 0;
}

